/**
@file temperatureplot.h
*/
#ifndef TEMPERATUREPLOT_H
#define TEMPERATUREPLOT_H

/**
@brief displays the temperature against time at various intervals
@brief temperature is in degrees celsius
@author Augustine K Kizito
@data April 2015
*/




class TemperaturePlot
{
private:
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen
    int xCord; // tracks the x coordinate
    char tempString[32]; // temperature value as string
    bool pause; // tracks if the plotting has been paused
    int pInterval; // determines the intervals at which data is plotted
    Ticker timer; // plots data at particular intervals

    void onBack(); // user pressed the back button
    void onEnter(); // user pressed the on enter button
    void plotData(); // plot temperature data
    void clearPlot(); // clear the plotted data
    void pausePlot(); // it pauses plotting of data so that the user can see it

public:
    /**
    Manages the execution of the Temperature Plot Screen 
    
    @param - integer interval, the intervals after which temperature is plotted against time on the LCD screen
    @returns
     -1   - navigate to previous screen
    
    */
    int start(int interval);

};

int TemperaturePlot::start(int interval)
{
    // Initialisations
    pInterval = interval;
    xCord = 0; // set the x cordinate to 0
    pause = false;

    backButton.mode(PullDown); // activate pullDown Resistor
    enterButton.mode(PullDown); // activate pullDown Resistor
    
    backButton.rise(this,&TemperaturePlot::onBack); // call onBack when the back button is pressed
    enterButton.rise(this,&TemperaturePlot::onEnter); // call onBack when the back button is pressed

    plotData(); //plot data onto the screen
    lcd.printString("Deg Cel", 40,1);

   // plot data on LCD screen depending on the interval previously selected by the user
    if ( interval == 0) {
        timer.attach(this, &TemperaturePlot::plotData,0.1); // call ISR every 1.0 second
    } else if ( interval == 1) {
        timer.attach(this, &TemperaturePlot::plotData,0.5); // call ISR every 5 seconds
    } else if ( interval == 2) {
        timer.attach(this, &TemperaturePlot::plotData,1); // call ISR every 10 seconds
    } else  {
        timer.attach(this, &TemperaturePlot::plotData,5); // call ISR every 30 seconds
    }

    changeScreen = false;
    debounce.start(); // start the debounce timer

    while (!changeScreen) {

        // mbed goes to sleep to save power
        Sleep();

    }
    
    // remove all interrupts
    backButton.rise(NULL);
    enterButton.rise(NULL);
    timer.detach();
    return nextScreen;
}

void TemperaturePlot::plotData()
{
    if (xCord > 83 ) {

        xCord = 0; // reset the xCord back to zero

        clearPlot(); // clear the plot region

    }

    float temperature = getTemperature(); // retreive temperature from the sensor

    lcd.drawLine(xCord,(31*((50-temperature)/50)+16),xCord,47,1);

    // print temperature onto the screen
    sprintf(tempString,"%0.2f",temperature);
    lcd.printString(tempString,0,1);

    // increment the coordinate
    xCord++;


}

void TemperaturePlot::onBack()
{
    if(debounceSuccess()) { // debouncing successful
        // soundt the buzze
        playSound();
        nextScreen = -1;
        changeScreen = true; // user wants to go to a different screen
        // reset debounce timer
        debounce.reset();
    }

}

void TemperaturePlot::clearPlot()
{
    // clears the plot area

    for ( int x = 0; x < 84; x++) {

        for ( int y = 16; y < 48; y++) {

            lcd.clearPixel(x,y);

        }

    }

    lcd.refresh();

}

void TemperaturePlot::onEnter()
{
    if (debounceSuccess()) { // debouncing successful

        playSound(); // play sound
        pausePlot(); // pause any plotting
        debounce.reset(); // reset debounce timer
    }

}

void TemperaturePlot::pausePlot()
{
    pause = !pause;

    if (pause) {
        timer.detach(); // detach timer and stop plotting data
    } else {

        clearPlot(); // clear the plot area
        xCord = 0; // start plotting from extreme left
        plotData();
        
        // start plotting data again
        if ( pInterval == 0) {
            timer.attach(this, &TemperaturePlot::plotData,0.1); // call ISR every 0.1 second
        } else if ( pInterval == 1) {
            timer.attach(this, &TemperaturePlot::plotData,0.5); // call ISR every 0.5 seconds
        } else if ( pInterval == 2) {
            timer.attach(this, &TemperaturePlot::plotData,1); // call ISR every 1 seconds
        } else  {
            timer.attach(this, &TemperaturePlot::plotData,5); // call ISR every 5 seconds
        }


    }
}

#endif