/**
@file temperaturelog.h
*/

#ifndef TEMPERATURELOG_H
#define TEMPERATURELOG_H

#include "Outputs.h"
#include "ConfigFile.h"
#include "Sensors.h"

/**
@brief logs temperature values over a predetermined length of time\n
@brief the logged data is then stored on the onboard flash memory\n
@brief the LCD screen is turned off after 5 seconds to save energy
@date April 2015
*/



class TemperatureLog
{
private:
    bool logFinished; // tracks if logging was successfuly finished
    int logCounter; //tracks how many times data has been logged
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen
    Timeout stop; // timeout that will turn off the screen after a given time
    Ticker logTimer; // ticker that will log data at intervals
    Ticker batLedTimer; // ticker that will check battery status at intervals
    ConfigFile tcfgW; // temperature CFG Write


    void onBack(); // user pressed the back button
    void turnOffScreen(); // powers down the lcd screen
    void logData(); // logs data by storing it in ConfigFile object
    void saveData(); // saves data to the mbed flash drive
    void initCfg(int duration); // initialises the cfg object with vital data
    void checkBatVoltage(); // checks if batteryVoltage is low

public:
    /**
    Manages the execution of the Temperature Log Screen

    @param - integer duration, the length of time to log temperature
    @returns
    -1   - navigate to previous screen

    */
    int start(int duration);


};

int TemperatureLog::start(int duration)
{
    // initialisations
    logFinished = false;
    logCounter = 0;
    lcd.printString("Logging...",20,2);

    backButton.mode(PullDown); // activate pullDown Resistor
    backButton.rise(this,&TemperatureLog::onBack); // call onBack when the back button is pressed

    stop.attach(this,&TemperatureLog::turnOffScreen, 5); // turn off LCD screen after 5 seconds
    batLedTimer.attach(this, &TemperatureLog::checkBatVoltage, 10); // check battery voltage  every 5 seconds

    // check battery voltage
    checkBatVoltage();
    // turn on the log led
    turnOnLogLed();
    // initialise the cfg object
    initCfg(duration);
    // start the debounce timer

    debounce.start();

    // 84 data points should be recorded for every log

    logData(); // log the data

    if ( duration == 0) {
        // this intervals results makes 84 logs in 5 minutes
        logTimer.attach(this, &TemperatureLog::logData,3.614);
    } else if ( duration == 1) {
        // this intervals results makes 84 logs in 10 minutes
        logTimer.attach(this, &TemperatureLog::logData,7.229);
    } else if ( duration == 2) {
        // this intervals results makes 84 logs in 30 minutes
        logTimer.attach(this, &TemperatureLog::logData,21.687);
    } else  {
        // this intervals results makes 84 logs in 1 hour
        logTimer.attach(this, &TemperatureLog::logData,42.373);
    }

    while (!logFinished) {

        // mbed goes to sleep to save power
        Sleep();

    }

    // remove interrupts
    logTimer.detach();
    backButton.rise(NULL);

    turnOffLogLed();

    return -1;
}

void TemperatureLog::initCfg( int duration )
{

    // Create the Duration Key and Value Pair
    char *theDurationKey = "durationKey";
    char theDurationValue[32];

    if ( duration == 0 ) {

        char theDurationString[32] = "5 min";
        strcpy(theDurationValue,theDurationString);

    } else if ( duration == 1) {

        char theDurationString[32] = "10 min";
        strcpy(theDurationValue,theDurationString);

    } else if ( duration == 2) {

        char theDurationString[32] = "30 min";
        strcpy(theDurationValue,theDurationString);

    } else {

        char theDurationString[32] = "1 hour";
        strcpy(theDurationValue,theDurationString);
    }

    // Create the Time and Value Pair
    char *theTimeKey = "timeKey";
    char theTimeValue[32];

    // Create the Date Key and Value Pair
    char *theDateKey = "dateKey";
    char theDateValue[32];

    ////GET THE CURRENT TIME///////////
    time_t seconds = time(NULL);
    strftime(theDateValue, 32, "%d/%m/%y", localtime(&seconds)); // format the date in day/month/year
    strftime(theTimeValue, 32, "%R", localtime(&seconds)); // format the time in HH:MM 24 Hr clock
    ////////////////////////////////////////////

    // set the created Key and Value Pairs to the cfg object
    tcfgW.setValue(theDateKey,theDateValue);
    tcfgW.setValue(theTimeKey,theTimeValue);
    tcfgW.setValue(theDurationKey,theDurationValue);


}

void TemperatureLog::logData()
{

    if ( logCounter < 84 ) { // // data is still logging

        // get the current temperature
        float temperature = getTemperature();

        // create a new Counter Key and Value Pair
        char *theCounterKey = "counterKey";
        char theCounterValue[32] = "";

        // assign the logCounter as the CounterValue
        sprintf(theCounterValue,"%d",logCounter);

        // Create a new Key and Value Pair
        char theKey[32] = "key";
        char theValue[32] = "";

        // concatentate the Key Value and Counter Value strings
        strcat(theKey,theCounterValue);

        sprintf(theValue,"%0.2f",temperature); // converting float to string

        // set the created key and value pairs to the cfg object
        tcfgW.setValue(theCounterKey,theCounterValue);
        tcfgW.setValue(theKey,theValue);


    } else {

        saveData(); // save the data
        logFinished = true; // logging has finished successfully
    }
    // increment the log counter
    logCounter++;

}

void TemperatureLog::saveData()
{
    // save the data to onboard flash memory as a cfg file
    tcfgW.write("/local/temp.cfg");

}

void TemperatureLog::onBack()
{
    if (debounceSuccess()) { // debouncing successful
        playSound();
        // if user presses the back button, save data and terminate logging

        saveData(); // save the data
        logFinished = true; // logging has finished successfully
        debounce.reset(); // reset the debounce timer
    }


}

void TemperatureLog::turnOffScreen()
{
    // turn off the LCD screen to save power
    lcd.turnOff();

}

void TemperatureLog::checkBatVoltage()
{
    // read the battery voltage
    float batteryVoltage = getVoltage();

    if (batteryVoltage < 6.5) {  // check if battery is low

        if(!batStatLedState) { // if the battery low led is not already on

            turnOnBatStatLed(); // turn on the battery low led
        }

    } else {

        turnOffBatStatLed(); // ensure that the battery low led is turned off

    }


}

#endif