/**
@file SettingsScreen.h

*/

#ifndef SETTINGSSCREEN_H
#define SETTINGSSCREEN_H

#include "ListController.h"
#include "Inputs.h"

/**
@brief Displays the settings on the LCD screen \n
@brief It is a list with three selectable options :\n
@brief brightness, sound fx and battery status\n
@brief brightness allows the user to set the LCD backlight brightness to \n
@brief 100%, 50%, 25% and Off\n
@brief sound fx allows the user to turn on or off the sound, i.e the buzzer\n
@brief both brightness and sound settings are persistent, i.e the preseved when the mbed is turned off\n
@brief battery status is only shows the current battery voltage and along with the sentiment\n
@author Augustine Kizito K
@date April 2015
*/



class SettingsScreen : public ListController // inherit the list Controller properties
{
private:
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen

    void onScrollUp(); // user pressed the scroll up button
    void onScrollDown(); // user pressed the scroll down button
    void onEnter(); // user pressed the enter button
    void onBack(); // user pressed the back button

public:
    /**
    Creates an instance of a Setttings Screen
    
    */
    SettingsScreen()
        :ListController( "Settings", // Title
                         "Brightness", // Option 1
                         "Sound FX", // Option 2
                         "Battery Status", // Option 3
                         "", // Option 4
                         3) // Number of Options
    {};
    /**
    Manages the execution of the Settings Screen
    
    @returns
    
    -1 - Navigate to previous screen\n
    0  - User selected option 1\n
    1  - User selected option 2\n
    2  - User selected option 3
    
    */
    int start(); 


};














int SettingsScreen::start()
{

    // Activate PullDown Resistors
    enterButton.mode(PullDown);
    upButton.mode(PullDown);
    downButton.mode(PullDown);
    backButton.mode(PullDown);

    upButton.rise(this,&SettingsScreen::onScrollUp); // call onScrollUp when the up button is pressed
    downButton.rise(this,&SettingsScreen::onScrollDown); // call onScrollDown when the down button is pressed
    enterButton.rise(this,&SettingsScreen::onEnter); // call onEnter when the enter button is pressed
    backButton.rise(this,&SettingsScreen::onBack); // call onBack when the back button is pressed

    ListController::showInfo();
    changeScreen = false;

    debounce.start(); // start the debounce timer

    while(!changeScreen) {

        // mbed goes to sleep to save power
        Sleep();
    }

    // detach all interrupts
    upButton.rise(NULL);
    downButton.rise(NULL);
    enterButton.rise(NULL);

    return nextScreen;
}

void SettingsScreen::onScrollUp()
{
    if (debounceSuccess()) {  // check if debouncing succeeded

        ListController::scrollUp(); // scroll up
        debounce.reset(); // reset the debounce timer
    }
}

void SettingsScreen::onScrollDown()
{
    if (debounceSuccess()) { // check if debouncing succeeded

        ListController::scrollDown(); // scroll down
        debounce.reset(); // reset the debounce timer
    }
}

void SettingsScreen::onEnter()
{
    if (debounceSuccess()) { // check if debouncing succeeded
        // sound the buzzer
        playSound();
        nextScreen = ListController::getCurrentOption(); //  get the option that user selected
        changeScreen = true; // user wants to go to a different screen
        debounce.reset(); // reset the debounce timer
    }
}

void SettingsScreen::onBack()
{
    if(debounceSuccess()) {
      // sound the buzzer
      playSound();
      nextScreen = -1; // go to previous screen
      changeScreen = true; // user wants to go to a different screen
      // reset the debounce timer
      debounce.reset();
    }

}

#endif