/**
@file PressureScreen.h

*/

#ifndef PRESSURESCREEN_H
#define PRESSURESCREEN_H

#include "ListController.h"
#include "Inputs.h"
#include "Outputs.h"

/**

@brief Displays the pressure options on the LCD screen \n
@brief It is a list with two selectable options :\n
@brief view prs and plot prs\n
@brief view prs shows the current pressure along with the sentiment\n
@brief prs plot plots the pressure against time on the LCD screen at various intervals\n
@author Augustine Kizito K
@date April 2015

*/



class PressureScreen : public ListController // inherit the list Controller properties
{
private:
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen

    void onScrollUp(); // user pressed the scroll up button
    void onScrollDown(); // user pressed the scroll down button
    void onEnter(); // user pressed the enter button
    void onBack(); // user pressed the back button

public:
    /**
    Creates an instance of the pressure screen
    
    */
    PressureScreen()
        :ListController( "Pressure", // Title
                         "View Prs", // Option 1
                         "Prs Plot", // Option 2
                         "", // Option 3
                         "", // Option 4
                         2) // Number of Options
    {};
    /**
    Manages the execution of the pressure screen
    
    @returns
    
    -1 - Navigate to previous screen\n
    0  - User selected option 1\n
    1  - User selected option 2
    */
    int start(); 


};

int PressureScreen::start()
{

    // Activate PullDown Resistors
    enterButton.mode(PullDown);
    upButton.mode(PullDown);
    downButton.mode(PullDown);
    backButton.mode(PullDown);

    upButton.rise(this,&PressureScreen::onScrollUp); // call onScrollUp when the up button is pressed
    downButton.rise(this,&PressureScreen::onScrollDown); // call onScrollDown when the down button is pressed
    enterButton.rise(this,&PressureScreen::onEnter); // call onEnter when the enter button is pressed
    backButton.rise(this,&PressureScreen::onBack); // call onBack when the back button is pressed

    // Launch the List Controller
    ListController::showInfo();
    changeScreen = false;

    debounce.start(); // start the debounce timer

    while(!changeScreen) {

        // mbed goes to sleep to save power
        Sleep();
    }

    // detach all interrupts
    upButton.rise(NULL);
    downButton.rise(NULL);
    enterButton.rise(NULL);
    backButton.rise(NULL);

    return nextScreen;
}

void PressureScreen::onScrollUp()
{
    if (debounceSuccess()) {  // check if debouncing succeeded

        ListController::scrollUp(); // scroll up
        debounce.reset(); // reset the debounce timer
    }
}

void PressureScreen::onScrollDown()
{
    if (debounceSuccess()) { // check if debouncing succeeded

        ListController::scrollDown(); // scroll down
        debounce.reset(); // reset the debounce timer
    }
}

void PressureScreen::onEnter()
{
    if (debounceSuccess()) { // check if debouncing succeeded
        // sound the buzzer
        playSound();
        nextScreen = ListController::getCurrentOption(); //  get the option that user selected
        changeScreen = true; // user wants to go to a different screen
        debounce.reset(); // reset the debounce timer
    }
}

void PressureScreen::onBack()
{
    if(debounceSuccess()) {
      // sound the buzzer
      playSound();
      nextScreen = -1; // go to previous screen
      changeScreen = true; // user wants to go to a different screen
      debounce.reset();
    }

}

#endif