/**
@file pressureplot.h
*/
#ifndef PRESSUREPLOT_H
#define PRESSUREPLOT_H

#include "Sensors.h"
/**
@brief displays the pressure against time at various intervals\n
@brief pressure is in mbars
@author Augustine K Kizito
@data April 2015
*/

class PressurePlot
{
private:
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen
    int xCord; // tracks the x coordinate
    char prsString[32]; // temperature value as string
    bool pause; // tracks if the plotting has been paused
    int pInterval; // determines the intervals at which data is plotted
    Ticker timer; // plots data at particular intervals

    void onBack(); // user pressed the back button
    void onEnter(); // user pressed the on enter button
    void plotData(); // plot temperature data
    void clearPlot(); // clear the plotted data
    void pausePlot(); // it pauses plotting of data so that the user can view it

public:
     /**
    Manages the execution of the Pressure Plot Screen 
    
    @param - integer interval, the intervals after which pressure is plotted against time on the LCD screen
    @returns
     -1   - navigate to previous screen
    
    */
    int start(int interval);

};

int PressurePlot::start(int interval)
{
    // initialisations
    pInterval = interval;
    xCord = 0; // set the x cordinate to 0
    pause = false;

    backButton.mode(PullDown); // activate pullDown Resistor
    enterButton.mode(PullDown); // activate pullDown Resistor
    
    backButton.rise(this,&PressurePlot::onBack); // call onBack when the back button is pressed
    enterButton.rise(this,&PressurePlot::onEnter); // call onBack when the back button is pressed

    plotData(); // plot data onto screen
    lcd.printString("mbars", 40,1);

    // plot data on LCD screen depending on the interval previously selected by the user
    if ( interval == 0) {
        timer.attach(this, &PressurePlot::plotData,0.1); // call ISR every 1.0 second
    } else if ( interval == 1) {
        timer.attach(this, &PressurePlot::plotData,0.5); // call ISR every 5 seconds
    } else if ( interval == 2) {
        timer.attach(this, &PressurePlot::plotData,1); // call ISR every 10 seconds
    } else  {
        timer.attach(this, &PressurePlot::plotData,5); // call ISR every 30 seconds
    }

    changeScreen = false;
    debounce.start(); // start the debounce timer

    while (!changeScreen) {

        // mbed goes to sleep to save power
        Sleep();

    }
    
    // remove all interrupts
    backButton.rise(NULL);
    enterButton.rise(NULL);
    timer.detach();
    return nextScreen;
}

void PressurePlot::plotData()
{
    if (xCord > 83 ) {

        xCord = 0; // reset the xCord back to zero

        clearPlot(); // clear the plot region

    }

    float pressure = getPressure(); // retreive pressure from the sensor

    lcd.drawLine(xCord,(31*((1200-pressure)/1200)+16),xCord,47,1);

    // show pressure on the screen
    sprintf(prsString,"%0.2f",pressure);
    lcd.printString(prsString,0,1);
 
    //increment the coordinate
    xCord++;


}

void PressurePlot::onBack()
{
    if(debounceSuccess()) { // debouncing successful
       // sound the buzzer
        playSound();
        nextScreen = -1;
        changeScreen = true; // user wants to go to a different screen
        // reset the debounce timer
        debounce.reset();
    }

}

void PressurePlot::clearPlot()
{
   // clears the plot area
    for ( int x = 0; x < 84; x++) {

        for ( int y = 16; y < 48; y++) {

            lcd.clearPixel(x,y);

        }

    }

    lcd.refresh();

}

void PressurePlot::onEnter()
{
    if (debounceSuccess()) { // debouncing successful

        playSound(); // play sound
        pausePlot(); // pause any plotting
        debounce.reset(); // reset debounce timer
    }

}

void PressurePlot::pausePlot()
{
    pause = !pause;

    if (pause) {
        timer.detach(); // detach timer and stop plotting data
    } else {

        clearPlot(); // clear the plot area
        xCord = 0; // start plotting from extreme left
        plotData();
        
        // start plotting data again
        if ( pInterval == 0) {
            timer.attach(this, &PressurePlot::plotData,0.1); // call ISR every 0.1 second
        } else if ( pInterval == 1) {
            timer.attach(this, &PressurePlot::plotData,0.5); // call ISR every 0.5 seconds
        } else if ( pInterval == 2) {
            timer.attach(this, &PressurePlot::plotData,1); // call ISR every 1 seconds
        } else  {
            timer.attach(this, &PressurePlot::plotData,5); // call ISR every 5 seconds
        }


    }
}

#endif