/**
@file pressurelog.h
*/
#ifndef PRESSURELOG_H
#define PRESSURELOG_H

#include "Outputs.h"
#include "ConfigFile.h"
#include "Sensors.h"

/**
@brief logs pressure values over a predetermined length of time\n
@brief the logged data is then stored on the onboard flash memory\n
@brief the lCD screen is turned off after 5 seconds to save energy
@author Augustine K Kizito
@date April 2015
*/


class PressureLog
{
private:
    bool logFinished; // tracks if logging was successfuly finished
    int logCounter; //tracks how many times data has been logged
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen
    Timeout stop; // timeout that will turn off the screen after a given time
    Ticker logTimer; // ticker that will log data at intervals
    Ticker batLedTimer; // ticker that will check battery status at intervals
    ConfigFile pcfgW; // pressure CFG object

    void onBack(); // user pressed the back button
    void turnOffScreen(); // powers down the lcd screen
    void logData(); // logs data by storing it in ConfigFile object
    void saveData(); // saves data to the mbed flash drive
    void initCfg(int duration); // initialises the cfg object with vital data
    void checkBatVoltage(); // checks if batteryVoltage is low

public:
    /**
    Manages the execution of the Pressure Log Screen 
    
    @param - integer duration, the length of time to pressure
    @returns
     -1   - navigate to previous screen
    
    */
    int start(int duration);


};

int PressureLog::start(int duration)
{
    // initialisations
    logFinished = false;
    logCounter = 0;
    lcd.printString("Logging...",20,2);

    backButton.mode(PullDown); // activate pullDown Resistor
    backButton.rise(this,&PressureLog::onBack); // call onBack when the back button is pressed

    stop.attach(this,&PressureLog::turnOffScreen, 5); // turn off LCD screen after 5 seconds
    batLedTimer.attach(this, &PressureLog::checkBatVoltage, 10); // check battery voltage  every 5 seconds

    // check the Battery Voltage
    checkBatVoltage();
    // Turn on the Log Led
    turnOnLogLed();
    // Initialise CFG object
    initCfg(duration);
    // start the debounce timer
    debounce.start();

    // 84 data points should be recorded for every log

    logData(); // log the data

    if ( duration == 0) {
        // this intervals  makes 84 logs in 5 minutes
        logTimer.attach(this, &PressureLog::logData,3.614);
    } else if ( duration == 1) {
        // this intervals makes 84 logs in 10 minutes
        logTimer.attach(this, &PressureLog::logData,7.229);
    } else if ( duration == 2) {
        // this intervals makes 84 logs in 30 minutes
        logTimer.attach(this, &PressureLog::logData,21.687);
    } else  {
        // this intervals  makes 84 logs in 1 hour
        logTimer.attach(this, &PressureLog::logData,42.373);
    }

    while (!logFinished) {

        // mbed goes to sleep to save power
        Sleep();

    }

    // remove interrupts
    logTimer.detach();
    backButton.rise(NULL);

    turnOffLogLed();

    return -1;
}

void PressureLog::initCfg( int duration )
{

    // Create the Duration Key and Value Pair
    char *theDurationKey = "durationKey";
    char theDurationValue[32];

   // Set the Duration Value appropriately
    if ( duration == 0 ) {

        char theDurationString[32] = "5 min";
        strcpy(theDurationValue,theDurationString);

    } else if ( duration == 1) {

        char theDurationString[32] = "10 min";
        strcpy(theDurationValue,theDurationString);

    } else if ( duration == 2) {

        char theDurationString[32] = "30 min";
        strcpy(theDurationValue,theDurationString);

    } else {

        char theDurationString[32] = "1 hour";
        strcpy(theDurationValue,theDurationString);
    }

    // Create the Time and Value Pair
    char *theTimeKey = "timeKey";
    char theTimeValue[32];

    // Create the Date Key and Value Pair
    char *theDateKey = "dateKey";
    char theDateValue[32];

    ////GET THE CURRENT TIME///////////
    time_t seconds = time(NULL);
    strftime(theDateValue, 32, "%d/%m/%y", localtime(&seconds)); // format the date in day/month/year
    strftime(theTimeValue, 32, "%R", localtime(&seconds)); // format the time in HH:MM 24 Hr clock
    ////////////////////////////////////////////

    // set the created Key and Value Pairs to the cfg object
    pcfgW.setValue(theDateKey,theDateValue);
    pcfgW.setValue(theTimeKey,theTimeValue);
    pcfgW.setValue(theDurationKey,theDurationValue);


}

void PressureLog::logData()
{

    if ( logCounter < 84 ) { // data is still logging
    
        // get the current pressure
        float pressure = getPressure();

        // create a new Counter Key and Value Pair
        char *theCounterKey = "counterKey";
        char theCounterValue[32] = "";
        
        // assign the logCounter as the CounterValue
        sprintf(theCounterValue,"%d",logCounter);

        // Create a new Key and Value Pair
        char theKey[32] = "key";
        char theValue[32] = "";

        // concatentate the Key Value and Counter Value strings
        strcat(theKey,theCounterValue);

        sprintf(theValue,"%0.2f",pressure); // converting float to string

        // set the created key and value pairs to the cfg object
        pcfgW.setValue(theCounterKey,theCounterValue);
        pcfgW.setValue(theKey,theValue);


    } else {

        saveData(); // save the data
        logFinished = true; // logging has finished successfully
    }

    // increment the log counter
    logCounter++;

}

void PressureLog::saveData()
{
    // save the data to onboard flash memory as a cfg file
    pcfgW.write("/local/prs.cfg");

}

void PressureLog::onBack()
{
    if (debounceSuccess()) { // debouncing successful
        
        playSound();
        // if user presses the back button, save data and terminate logging

        saveData(); // save the data
        logFinished = true; // logging has finished successfully
        debounce.reset(); // reset the debounce timer

    }

}

void PressureLog::turnOffScreen()
{
    // turn off the LCD screen to save power
    lcd.turnOff();

}

void PressureLog::checkBatVoltage()
{
    // read the battery voltage
    float batteryVoltage = getVoltage();

    if (batteryVoltage < 6.5) { // check if battery is low

        if(!batStatLedState) { // if the battery low led is not already on

            turnOnBatStatLed(); // turn on the battery low led
        }


    } else {

        turnOffBatStatLed(); // ensure that the battery low led is turned off

    }


}

#endif