/**
@file LogScreen.h


*/

#ifndef LOGSCREEN_H
#define LOGSCREEN_H

#include "ListController.h"
#include "Inputs.h"

/**
@brief Displays the options related to persistent data on the LCD screen \n
@brief It is a list with two selectable options :\n
@brief new log and view log\n
@brief new log option allows the user to log temperature or pressure or a preset amount of time\n
@brief and store the recorded data on the mbed flash drive\n
@brief view log option allows the user to view data that has been previously logged, either temperature or pressure.\n
@brief user will be able to know, the date, time and duration of the log. the user will view a summary of the logged data\n
@brief i.e maximum, minimum and average values of the recorded data. A plot agains time of the data will also be available
@author Augustine Kizito K
@date April 2015
*/



class LogScreen : public ListController // inherit the list Controller properties
{
private:
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen

    void onScrollUp(); // user pressed the scroll up button
    void onScrollDown(); // user pressed the scroll down button
    void onEnter(); // user pressed the enter button
    void onBack(); // user pressed the back button

public:
    /**
    Creates an instance of the Log Screen
    
    */
    LogScreen()
        :ListController( "Log", // Title
                         "New Log", // Option 1
                         "View Log", // Option 2
                         "", // Option 3
                         "", // Option 4
                         2) // Number of Options
    {};
    /**
    Manages the execution of the Log Screen
    
    @returns
    
    -1 - Navigate to previous screen\n
    0  - User selected option 1\n
    1  - User selected option 2
    
    */
    int start(); // Manages the execution of the Temperature Screen


};

int LogScreen::start()
{

    // Activate PullDown Resistors
    enterButton.mode(PullDown);
    upButton.mode(PullDown);
    downButton.mode(PullDown);
    backButton.mode(PullDown);

    upButton.rise(this,&LogScreen::onScrollUp); // call onScrollUp when the up button is pressed
    downButton.rise(this,&LogScreen::onScrollDown); // call onScrollDown when the down button is pressed
    enterButton.rise(this,&LogScreen::onEnter); // call onEnter when the enter button is pressed
    backButton.rise(this,&LogScreen::onBack); // call onBack when the back button is pressed
    
    // Launch the List Controller
    ListController::showInfo();
    changeScreen = false;

    debounce.start(); // start the debounce timer

    while(!changeScreen) {

        // mbed goes to sleep to save power
        Sleep();
    }

    // detach all interrupts
    upButton.rise(NULL);
    downButton.rise(NULL);
    enterButton.rise(NULL);
    backButton.rise(NULL);

    return nextScreen;
}

void LogScreen::onScrollUp()
{
    if (debounceSuccess()) {  // check if debouncing succeeded

        ListController::scrollUp(); // scroll up
        debounce.reset(); // reset the debounce timer
    }
}

void LogScreen::onScrollDown()
{
    if (debounceSuccess()) { // check if debouncing succeeded

        ListController::scrollDown(); // scroll down
        debounce.reset(); // reset the debounce timer
    }
}

void LogScreen::onEnter()
{
    if (debounceSuccess()) { // check if debouncing succeeded
        // sound the buzzer
        playSound();
        nextScreen = ListController::getCurrentOption(); //  get the option that user selected
        changeScreen = true; // user wants to go to a different screen
        debounce.reset(); // reset the debounce timer
    }
}

void LogScreen::onBack()
{
    if(debounceSuccess()) { // debugging was successful
      // sound the buzzer
      playSound();
      nextScreen = -1; // go to previous screen
      changeScreen = true; // user wants to go to a different screen
      debounce.reset(); // reset the debounce timer
    }

}

#endif