/**
@file ListController.h

*/

#ifndef LISTCONTROLLER_H
#define LISTCONTROLLER_H

#include "mbed.h"
#include "Outputs.h"

/**
@brief Manages the display of upto 4 possible options on LCD that the user can select.\n
@brief also has a scrolling feature that allows user to browse the options\n
@author Augustine Kizito K
@date April 2015
*/
class ListController
{
private:
    char title[32]; // title of the list
    char option1[32]; // option 1 on the list
    char option2[32]; // option 2 on the list
    char option3[32]; // option 3 on the list
    char option4[32]; // option 4 on the list
    int numberOfOptions; // available number of options
    int currentOption; // tracks position of current cell

    void invertPixels(int option); // inverts pixels in a given bank i.e highlighting an option

protected:
    /**
    Creates a List Controller instance
    
    @param t - string title of list
    @param op1 - option 1 on the list
    @param op2 - option 2 on the list
    @param op3 - option 3 on the list
    @param op4 - option 4 on the list
    @param nOO 0 - available number of options
    
    */
    ListController( char t[], char op1[], char op2[], char op3[], char op4[], int nOO); 
    
    /**
    Populates the LCD screen with the a list of browse-able and selectable options
    
    */
    void showInfo(); // populate the screen with controller data
    
    /**
    Scrolls to the previous option
    
    */
    void scrollUp(); 
    
    /**
    Scrolls to the next option
    
    */
    void scrollDown();
    
    /**
    Gets the current/highlighted option and returns is to the calling function
    
    @returns the currentt option 0 - 3
    */
    int getCurrentOption(); // returns the highlighted Option 0-3

};







ListController::ListController( char t[], char op1[], char op2[], char op3[], char op4[],int nOO)
{
    // initialisations
    strcpy(title,t);
    strcpy(option1,op1);
    strcpy(option2,op2);
    strcpy(option3,op3);
    strcpy(option4,op4);
    numberOfOptions = nOO;
    currentOption = 0;  

}

void ListController::showInfo()
{
    lcd.printString(title,15,1); //  print title
    lcd.printString(option1,0,2);  // print option1
    lcd.printString(option2,0,3);  // print option2
    lcd.printString(option3,0,4); // print option3
    lcd.printString(option4,0,5); // print option4
    
    invertPixels(currentOption); // highlight the current option
}

void ListController::scrollUp()
{
    playSound(); // buzzer makes sound
    invertPixels(currentOption); // UN invert/highlight current option
    currentOption--;  // current position is less by 1

    if (currentOption < 0) { // prevents from inverting a non existent option

        currentOption = 0;
        invertPixels(currentOption); // invert all pixels in current bank

    } else {
        invertPixels(currentOption); // invert all pixels in current bank
    }


}

void ListController::scrollDown()
{
    playSound(); // buzzer makes sound
    invertPixels(currentOption); // UN-invert/highlight the current option
    currentOption++; // current option is more by 1

    if (currentOption > numberOfOptions-1) { // prevents from inverting a non existant option

        currentOption = numberOfOptions-1;
        invertPixels(currentOption); // invert all pixels in current option
    } else {
        invertPixels(currentOption); // invert all pixels in current option
    }
}


int ListController::getCurrentOption()
{ 
    return currentOption;

}

void ListController::invertPixels(int option)
{


    int onset; // x coordinate where inversion starts
   
   // first bank consists of status bar showing time and battery icon
   // second bank consists of the title
   
     if (option == 0) { // invert third bank
        onset = 15;
    } else if ( option == 1) { // invert fourth bank
        onset = 23;
    } else if ( option == 2) { // invert fifth bank
        onset = 31;
    } else {
        onset = 39;  // invert sixth bank
    }

    int termination = onset + 9; // bank's length height is 9 pixels

    for ( int m = onset; m < termination; m++) {

        for ( int n = 0; n < 84; n++) {

            if((lcd.getPixel(n,m)) == 0) { // if the pixel is clear
                lcd.setPixel(n,m);  // set the pixel
            } else {
                lcd.clearPixel(n,m); // else clear the pixel
            }

        }
    }

    lcd.refresh(); // refresh the lcd screen

}

#endif