/**
@file DurationScreen.h

*/

#ifndef DURATIONSCREEN_H
#define DURATIONSCREEN_H

#include "ListController.h"
#include "Inputs.h"

/**
@brief Allows the user to select different time lengths over which they want to log data\n
@brief Consists of a List controller with 4 selectable options:\n
@brief 5 minutes, 10 minutes, 30 minutes, 1 hour\n
@author Augustine K Kizito\n
@date April 2015
*/


class DurationScreen : public ListController // inherit the list Controller properties
{
private:
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen

    void onScrollUp(); // user pressed the scroll up button
    void onScrollDown(); // user pressed the scroll down button
    void onEnter(); // user pressed the enter button
    void onBack(); // user pressed the back button

public:
    /**
    This function creates an instance of the Duration Screen
    
    */
    DurationScreen()
        :ListController( "Duration", // Title
                         "5 minutes", // Option 1
                         "10 minutes", // Option 2
                         "30 minutes", // Option 3
                         "1 hour", // Option 4
                         4) // Number of Options
    {};
    /**
    Manages the execution of the Duration Screen
    
    @returns
    
    -1 - Navigate to previous screen\n
    0  - User selected option 1\n
    1  - User selected option 2\n
    2  - User selected option 3\n
    3  - User selected option 4
    
    */
    int start(); 


};



















int DurationScreen::start()
{

    // Activate PullDown Resistors
    enterButton.mode(PullDown);
    upButton.mode(PullDown);
    downButton.mode(PullDown);
    backButton.mode(PullDown);

    upButton.rise(this,&DurationScreen::onScrollUp); // call onScrollUp when the up button is pressed
    downButton.rise(this,&DurationScreen::onScrollDown); // call onScrollDown when the down button is pressed
    enterButton.rise(this,&DurationScreen::onEnter); // call onEnter when the enter button is pressed
    backButton.rise(this,&DurationScreen::onBack); // call onBack when the back button is pressed

    // Launch the List Controller
    ListController::showInfo();
    changeScreen = false;

    debounce.start(); // start the debounce timer

    while(!changeScreen) {

        // mbed goes to sleep to save power
        Sleep();
    }

    // detach all interrupts
    upButton.rise(NULL);
    downButton.rise(NULL);
    enterButton.rise(NULL);
    backButton.rise(NULL);

    return nextScreen;
}

void DurationScreen::onScrollUp()
{
    if (debounceSuccess()) {  // check if debouncing succeeded

        ListController::scrollUp(); // scroll up
        debounce.reset(); // reset the debounce timer
    }
}

void DurationScreen::onScrollDown()
{
    if (debounceSuccess()) { // check if debouncing succeeded

        ListController::scrollDown(); // scroll down
        debounce.reset(); // reset the debounce timer
    }
}

void DurationScreen::onEnter()
{
    if (debounceSuccess()) { // check if debouncing succeeded
       // sound the buzzer
        playSound();
        nextScreen = ListController::getCurrentOption(); //  get the option that user selected
        changeScreen = true; // user wants to go to a different screen
        debounce.reset(); // reset the debounce timer
    }
}

void DurationScreen::onBack()
{
    if(debounceSuccess()) { // debouncing was successful
      // sound the buzzer
      playSound();
      nextScreen = -1; // go to previous screen
      changeScreen = true; // user wants to go to a different screen
      debounce.reset();
    }

}

#endif