/**
@file BrightnessScreen.h


*/
#ifndef BRIGHTNESSSCREEN_H
#define BRIGHTNESSSCREEN_H

#include "SettingController.h"
#include "Inputs.h"


/**

@brief Shows the current brightness setting of the LCD backlight brightness\n
@brief Consists of a Setting Controller\n
@brief brightness can be set to 100%, 50%, 25%, or Off\n
@brief the setting is persistent i.e saved onto onboard flash memory\n
@author Augustine K Kizito\n
@date April 2015

*/



class BrightnessScreen: public SettingController // inherit SettingController class
{
private:
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    char *brightnessKey; // key to be used for the cfg file
    char brightnessValue[BUFSIZ]; // buffer to store the brightness value retrieved from cfg file
    int brightness; // the brightness stored as an int

    void onScrollUp(); // user pressed the scroll up button
    void onScrollDown(); // user pressed the scroll down button
    void onEnter(); // user pressed the enter button
    void onBack(); // user pressed the back button
    void saveSetting(); // save a selected option as a setting
    void adjustBrightness(); //adjust brightness according to the stored setting

public:
    /**
    This function creates an instance of the brightness screen
    
    */
    BrightnessScreen()
        :SettingController( "Brightness", // Title
                            "100%", // Option 1
                            "50%", // Option 2
                            "25%", // Option 3
                            "Off", // Option 4
                            4) // Number of Options
    {};
    /**
    This function manages the exectution of the brightness screen
    
    @returns
     -1 - navigate to previous screen
    */
    int start();

};



int BrightnessScreen::start()
{

    enterButton.mode(PullDown); // activate pull down resistor
    upButton.mode(PullDown); // activate pull down resistor
    downButton.mode(PullDown); // acivate pull down resistor
    backButton.mode(PullDown); // activate pull down resistor

    upButton.rise(this,&BrightnessScreen::onScrollUp); // call onScrollUp when the up button is pressed
    downButton.rise(this,&BrightnessScreen::onScrollDown); // call onScrollDown when the down button is pressed
    enterButton.rise(this,&BrightnessScreen::onEnter); // call onEnter when the enter button is pressed
    backButton.rise(this, &BrightnessScreen::onBack);

    brightnessKey = "brightness ";

    // prepare the cfg file to be read
    cfg.read("/local/bright.cfg");

    // if key aand value exist, retrieve and store
    if (cfg.getValue(brightnessKey, &brightnessValue[0], sizeof(brightnessValue))) { // if key aand value exist, retrieve and store

        brightness = atoi(brightnessValue); // convert string to integer
        currentBrightness = brightness;
        // set the ListController setting appropriateley
        if (brightness == 100) {
            SettingController::setCurrentSetting(0);
        } else if (brightness == 50) {
            SettingController::setCurrentSetting(1);
        } else if (brightness == 25) {
            SettingController::setCurrentSetting(2);
        } else {
            SettingController::setCurrentSetting(3);
        }
    }

    adjustBrightness(); // adjust brightness apprpriately
    SettingController::showInfo(); //populate the settingController
    changeScreen = false; // stay on this screen until further notice

    while(!changeScreen) {
        // mbed goes to sleep to save power
        Sleep();
    }

    // detach all interrupts
    enterButton.rise(NULL);
    backButton.rise(NULL);

    return -1; // go to the previous screen

}

void BrightnessScreen::onEnter()
{
    if (debounceSuccess()) { // debouncing successful
        //sound the buzzer
        playSound();
        // save the selected option as the new setting
        saveSetting();

        //change the setting marker on the screen accordingly
        SettingController::changeSetting();

        //adjust the brigtness accordingly
        adjustBrightness();


        // reset the debounce timer
        debounce.reset();
    }

}

void BrightnessScreen::onBack()
{
    
    if (debounceSuccess()) { // debouncing successful
        // sound the buzzer
        playSound();
        changeScreen = true; // user wants to change screen

        // reset the debounce timer
        debounce.reset();
    }


}

void BrightnessScreen::onScrollDown()
{
    if (debounceSuccess()) { // debouncing successful

        SettingController::scrollDown(); //scroll Up
        // reset the timer
        debounce.reset();
    }

}

void BrightnessScreen::onScrollUp()
{
    if (debounceSuccess()) { // debouncing successful

        SettingController::scrollUp(); //scroll Down
        // reset the debounce timer
        debounce.reset();
    }

}

void BrightnessScreen::saveSetting()
{
    // retrieve the current option
    int currentOption = SettingController::getCurrentOption();



    if ( currentOption == 0) {
        cfg.setValue(brightnessKey,"100");  //100%
    } else if ( currentOption == 1) {
        cfg.setValue(brightnessKey,"50");  //50%
    } else if ( currentOption == 2) {
        cfg.setValue(brightnessKey,"25");  //25%
    } else if ( currentOption == 3) {
        cfg.setValue(brightnessKey,"0");  //Off
    } else {
        // do nothing
    }

    // write data to the file
    cfg.write("/local/bright.cfg");

}

void BrightnessScreen::adjustBrightness()
{

    // retireve currentSetting
    int currentSetting = SettingController::getCurrentSetting();

    if ( currentSetting == 0) {  // 100%
        currentBrightness = 100;
        lcd.setBrightness(1.0);
    } else if ( currentSetting == 1) { // 50%
        currentBrightness = 50;
        lcd.setBrightness(0.5);
    } else if ( currentSetting == 2) { // 25%
        currentBrightness = 25;
        lcd.setBrightness(0.25);
    } else if ( currentSetting == 3) { // Off
        currentBrightness = 0;
        lcd.setBrightness(0);
    } else {
        // do nothing
    }

}

#endif
