/**
@file SoundScreen.h

*/

#ifndef SOUNDSCREEN_H
#define SOUNDSCREEN_H

#include "SettingController.h"
#include "Inputs.h"
#include "Outputs.h"
#include "ConfigFile.h"

/**
@brief Shows the current sound setting of the buzzer\n
@brief Consists of a Setting Controller\n
@brief Sound can be set On or Off\n
@brief the setting is persistent i.e saved onto onboard flash memory\n
@author Augustine K Kizito
@date April 2015
*/



class SoundScreen: public SettingController // inherit SettingController class
{
private:
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    char *soundKey; // key to be used for the cfg file
    char soundValue[BUFSIZ]; // buffer to store the brightness value retrieved from cfg file
    int sound; // the brightness stored as an int

    void onScrollUp(); // user pressed the scroll up button
    void onScrollDown(); // user pressed the scroll down button
    void onEnter(); // user pressed the enter button
    void onBack(); // user pressed the back button
    void saveSetting(); // save a selected option as a setting
    void adjustSound(); //adjust brightness according to the stored setting

public:
    /**
    Creates an instance of the Sound Screen
    
    */
    SoundScreen()
        :SettingController( "Sound Fx", // Title
                            "On", // Option 1
                            "Off", // Option 2
                            "",
                            "",
                            2) // Number of Options
    {};
    /**
    Manages the execution of the Sound Screen
    
    @returns
     
     -1 - Navigate to previous screen
     
    */
    int start();

};




















int SoundScreen::start()
{

    enterButton.mode(PullDown); // activate pull down resistor
    upButton.mode(PullDown); // activate pull down resistor
    downButton.mode(PullDown); // acivate pull down resistor
    backButton.mode(PullDown); // activate pull down resistor

    upButton.rise(this,&SoundScreen::onScrollUp); // call onScrollUp when the up button is pressed
    downButton.rise(this,&SoundScreen::onScrollDown); // call onScrollDown when the down button is pressed
    enterButton.rise(this,&SoundScreen::onEnter); // call onEnter when the enter button is pressed
    backButton.rise(this, &SoundScreen::onBack);

    debounce.start(); // start the debounce timer

    soundKey = "sound      ";
    cfg.read("/local/sound.cfg"); // prepare the cfg file to be read

    // if key aand value exist, retrieve and store
    if (cfg.getValue(soundKey, &soundValue[0], sizeof(soundValue))) { // if key aand value exist, retrieve and store

        sound = atoi(soundValue); // convert string to integer

        // set the ListController setting appropriateley
        if (sound == 1) {
            SettingController::setCurrentSetting(0);
        } else {
            SettingController::setCurrentSetting(1);
        }
    }

    adjustSound(); // adjust brightness apprpriately
    SettingController::showInfo(); //populate the settingController
    changeScreen = false; // stay on this screen until further notice

    while(!changeScreen) {
        // mbed goes to sleep to save power
        Sleep();
    }

    // detach all interrupts
    enterButton.rise(NULL);
    backButton.rise(NULL);

    return -1; // go to the previous screen

}

void SoundScreen::onEnter()
{
    if (debounceSuccess()) { // debouncing successful

        // save the setting
        saveSetting();

        //change the setting marker on the screen accordingly
        SettingController::changeSetting();

        //adjust the sound accordingly
        adjustSound();

        playSound();
        // reset the debounce timer
        debounce.reset();
    }

}

void SoundScreen::onBack()
{
    if(debounceSuccess()) { // debounce successful
        // play the sound
        playSound();
        changeScreen = true; // user wants to change screen
        // reset the debounce timer
        debounce.reset();

    }


}

void SoundScreen::onScrollDown()
{
    if (debounceSuccess()) { // debouncing successful
        SettingController::scrollDown(); //scroll Up
        debounce.reset(); // reset the debounce timer
    }

}

void SoundScreen::onScrollUp()
{
    if (debounceSuccess()) { // debouncing was successful
        SettingController::scrollUp(); //scroll Down
        debounce.reset(); // reset the debounce timer
    }

}

void SoundScreen::saveSetting()
{
    // retrieve the current option
    int currentOption = SettingController::getCurrentOption();

    if ( currentOption == 0) {
        cfg.setValue(soundKey,"1");  //On
    } else if ( currentOption == 1) {
        cfg.setValue(soundKey,"0");  //Off
    } else {
        // do nothing
    }

    // write to the cfg file
    cfg.write("/local/sound.cfg");
}

void SoundScreen::adjustSound()
{

    // retireve currentSetting
    int currentSetting = SettingController::getCurrentSetting();

    if ( currentSetting == 0) {  // On
        soundFx = true;
    } else if ( currentSetting == 1) { // Off
        soundFx = false;
    } else {
        // do nothing
    }

}

#endif
