/**
@file IntervalScreen.h

*/

#ifndef INTERVALSCREEN_H
#define INTERVALSCREEN_H

#include "ListController.h"
#include "Inputs.h"

/**
@brief Allows the user to select different time intervals overwhich they would like to plot\n
@brief either temperature or pressure against time\n
@brief Consists of a List controller with 4 selectable options:\n
@brief 100ms, 500ms, 1 second, 5 seconds
@author Augustine K Kizito
@date April 2015
*/


class IntervalScreen : public ListController // inherit the list Controller properties
{
private:
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen

    void onScrollUp(); // user pressed the scroll up button
    void onScrollDown(); // user pressed the scroll down button
    void onEnter(); // user pressed the enter button
    void onBack(); // user pressed the back button

public:
    /**
    Creates an instance of Interval Screen
    
    */
    IntervalScreen()
        :ListController( "Interval", // Title
                         "100ms", // Option 1
                         "500ms", // Option 2
                         "1 second", // Option 3
                         "5 seconds", // Option 4
                         4) // Number of Options
    {};
    
    /**
    Manages the execution of the Interval Screen
    
    @returns
    
    -1 - Navigate to previous screen\n
    0  - User selected option 1\n
    1  - User selected option 2\n
    2  - User selected option 3\n
    3  - User selected option 4\n
    
    */
    int start(); 

};















int IntervalScreen::start()
{

    // Activate PullDown Resistors
    enterButton.mode(PullDown);
    upButton.mode(PullDown);
    downButton.mode(PullDown);
    backButton.mode(PullDown);

    upButton.rise(this,&IntervalScreen::onScrollUp); // call onScrollUp when the up button is pressed
    downButton.rise(this,&IntervalScreen::onScrollDown); // call onScrollDown when the down button is pressed
    enterButton.rise(this,&IntervalScreen::onEnter); // call onEnter when the enter button is pressed
    backButton.rise(this,&IntervalScreen::onBack); // call onBack when the back button is pressed

    ListController::showInfo(); // launch the List Controller
    changeScreen = false;

    debounce.start(); // start the debounce timer

    while(!changeScreen) {

        // mbed goes to sleep to save power
        Sleep();
    }

    // detach all interrupts
    upButton.rise(NULL);
    downButton.rise(NULL);
    enterButton.rise(NULL);
    backButton.rise(NULL);

    return nextScreen;
}

void IntervalScreen::onScrollUp()
{
    if (debounceSuccess()) {  // check if debouncing succeeded

        ListController::scrollUp(); // scroll up
        debounce.reset(); // reset the debounce timer
    }
}

void IntervalScreen::onScrollDown()
{
    if (debounceSuccess()) { // check if debouncing succeeded

        ListController::scrollDown(); // scroll down
        debounce.reset(); // reset the debounce timer
    }
}

void IntervalScreen::onEnter()
{
    if (debounceSuccess()) { // check if debouncing succeeded
        // sound the buzzer
        playSound();
        nextScreen = ListController::getCurrentOption(); //  get the option that user selected
        changeScreen = true; // user wants to go to a different screen
        debounce.reset(); // reset the debounce timer
    }
}

void IntervalScreen::onBack()
{
    if(debounceSuccess()) {
      // sound the buzzer
      playSound();
      nextScreen = -1; // go to previous screen
      changeScreen = true; // user wants to go to a different screen
      debounce.reset(); // reset the debounce timer
    }

}

#endif