/**
@file CurrentTemperatureScreen.h
 


*/
#ifndef CURRENTTEMPERATURESCREEN_H
#define CURRENTTEMPERATURESCREEN_H

#include "ParameterController.h"
#include "Inputs.h"
#include "Sensors.h"

/**
@brief Displays the current temperature along with it's sentiment on the LCD screen\n
@brief It consists of a parameter controller\n 
@brief temperature is shown in deg cel by default but can be toggled to deg fahr\n
@brief whent the enter button is pressed\n
@author Augustine Kizito K\n
@date April 2015
*/




class CurrentTemperatureScreen : public ParameterController // inherit ParameterController Properties
{
private:
    void setTemperatureSentiment( float temperature); // get temperature sentiment
    void temperatureUpdate();  // update the temperature
    float getCurrentTemperature(); // get the current temperature
    void changeUnits(); // change units form celsius to fahrenheit
    void onBack(); // user pressed the back button
    void onEnter(); // user pressed the enter button

    bool changeScreen; // tracks if the user pressed a button that changes screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen
    bool celsius; // tracks the current units of temperature
    Ticker tempTicker; // create ticker object


public:
    /**
    This function manages the execution of the temperature screen
    
    @ returns
     -1 - navigate to previous screen
    */
    int start();
    /**
    This function creates an instance of the current temperature scren
    
    */
    CurrentTemperatureScreen()
        :ParameterController( "Temperature", // Parameter
                              "Deg Cel") // Units
    {} // initialises parameter controller

};

int CurrentTemperatureScreen::start()
{

    changeScreen = false;
    celsius = true; // units are in celsius

    float currentTemperature = getCurrentTemperature(); // get current temperature

    backButton.mode(PullDown); // activate pullDown resistor
    enterButton.mode(PullDown); // activate pullDown resistor

    backButton.rise(this,&CurrentTemperatureScreen::onBack); // call onBack when the back button is pressed
    enterButton.rise(this,&CurrentTemperatureScreen::onEnter); // call onBack when the back button is pressed

    debounce.start(); // start the debouncing timer

    tempTicker.attach(this, &CurrentTemperatureScreen::temperatureUpdate, 10); // update parameter every 30 seconds

    // initialise the ParameterController appropriately
    ParameterController::setValue(currentTemperature);
    setTemperatureSentiment(currentTemperature);
    
    // Launch the Parameter Controller
    ParameterController::showInfo();

    while(changeScreen == false) { // shows the screen until the user presses the back button
        // mbed goes to sleep to save power
        Sleep();
    }

    // detach interrupts
    backButton.rise(NULL);
    enterButton.rise(NULL);
    tempTicker.detach();

    return nextScreen;
}

void CurrentTemperatureScreen::temperatureUpdate()
{

    float currentTemperature = getCurrentTemperature(); // get current temperature
    setTemperatureSentiment(currentTemperature); // update the temperature sentiment

    if (!celsius) { // units are in celsius
        //convert to fahrenheit
        currentTemperature = (currentTemperature*9/5)+32;
    }

    // initialise the ParameterController appropriately
    ParameterController::setValue(currentTemperature);
    ParameterController::updateInfo();

}


void CurrentTemperatureScreen::setTemperatureSentiment(float temperature)
{

    if (temperature > 35) { // Blazing if temperature is greature than 35 Deg
        ParameterController::setSentiment("Blazing   ");
    } else if ( temperature <= 35 && temperature > 25 ) {
        ParameterController::setSentiment("Hot       "); // Hot if temperature is between 35 and 25 Deg Cel
    } else if ( temperature <= 25 && temperature > 20 ) {
        ParameterController::setSentiment("Warm      "); // Warm if temperature is between 25 and 20 Deg Cel
    } else if ( temperature <= 20 && temperature > 15 ) {
        ParameterController::setSentiment("Cool      "); // Cool if temperature is between 20 and 15 Deg Cel
    } else if ( temperature <= 15 && temperature >= 10 ) {
        ParameterController::setSentiment("Cold      "); // Cold if temperature is between 15 and 10 Deg Cel
    } else {
        ParameterController::setSentiment("Frigid    "); // Frigid if temperature is less than 10
    }

}

float CurrentTemperatureScreen::getCurrentTemperature()
{
    float temperature = getTemperature(); // read temperature from BMP180 sensor
    return temperature;
}

void CurrentTemperatureScreen::onBack()
{
    if (debounceSuccess()) { // debugging successful
        playSound(); // play sound from buzzer
        nextScreen = -1;
        changeScreen = true; // user wants to view a different screen
        debounce.reset(); // reset the timer
    }


}

void CurrentTemperatureScreen::onEnter()
{
    if (debounceSuccess()) { // debugging successful
        playSound(); // sound the buzzer
        changeUnits(); // change the units
        debounce.reset(); // reset the timer
    }

}

void CurrentTemperatureScreen::changeUnits()
{
    // toggle the units form Cel to Fahr and back
    celsius = !celsius;

    float currentTemperature = getCurrentTemperature(); // get current temperature
    setTemperatureSentiment(currentTemperature);

    if (celsius) { // Units are in celsius
        ParameterController::setUnit("Deg Cel ");
    } else { // Units are in Fahrenheit
        ParameterController::setUnit("Deg Fahr");
        //convert to fahrenheit
        currentTemperature = (currentTemperature*9/5)+32;

    }

    // update the ParameterController appropriately
    ParameterController::setValue(currentTemperature);
    ParameterController::updateInfo();

}



#endif