/**
@file CurrentPressureScreen.h


*/


#ifndef CURRENTPRESSURESCREEN_H
#define CURRENTPRESSURESCREEN_H

#include "ParameterController.h"
#include "Inputs.h"
#include "Sensors.h"

/**
@brief Displays the current pressure along with it's sentiment on the LCD screen\n
@brief It consists of a parameter controller\n
@brief pressure is shown in mbars by default but can be toggled to kPscl\n
@brief when the enter button is pressed \n
@author Augustine Kizito K \n
@date April 2015 \n

*/


class CurrentPressureScreen : public ParameterController // inherit ParameterController Properties
{
private:
    void setPressureSentiment( float pressure); // get pressure sentiment
    void pressureUpdate();  // update the pressure
    float getCurrentPressure(); // get the current temperature
    void changeUnits(); // change units form mbar to Pa
    void onBack(); // user pressed the back button
    void onEnter(); // user pressed the enter button

    bool changeScreen; // tracks if the user pressed a button that changes screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen
    bool mbar; // tracks the current units of pressure
    Ticker prsTicker; // create ticker object


public:
    /**
    This function manages the execution of the current pressure screen
    
    */
    int start(); 
    /**
    Creates an instance of the Current Pressure Screen
    
    returns
    -1 - navigate to previous screen
    */
    CurrentPressureScreen()
        :ParameterController( "Pressure", // Parameter
                              "mbars") // Units
    {} // initialises parameter controller

};

int CurrentPressureScreen::start()
{

    changeScreen = false;
    mbar = true; // units are in millibars

    float currentPressure = getCurrentPressure(); // get current temperature

    backButton.mode(PullDown); // activate pullDown resistor
    enterButton.mode(PullDown); // activate pullDown resistor

    backButton.rise(this,&CurrentPressureScreen::onBack); // call onBack when the back button is pressed
    enterButton.rise(this,&CurrentPressureScreen::onEnter); // call onBack when the back button is pressed

    debounce.start(); // start the debouncing timer

    prsTicker.attach(this, &CurrentPressureScreen::pressureUpdate, 10); // update parameter every 30 seconds

    // initialise the ParameterController appropriately
    ParameterController::setValue(currentPressure);
    setPressureSentiment(currentPressure);
    
    // Launch the Parameter Controller
    ParameterController::showInfo();

    while(!changeScreen) { // shows the menu screen until the user presses the back button
        // mbed goes to sleep to save power
        Sleep();
    }

    // detach interrupts
    backButton.rise(NULL);
    enterButton.rise(NULL);
    prsTicker.detach();

    return nextScreen;
}

void CurrentPressureScreen::pressureUpdate()
{

    float currentPressure = getCurrentPressure(); // get current temperature
    // update the pressure sentiment
    setPressureSentiment(currentPressure);

    if (!mbar) { // units are currently is Pascals
        //convert to Pascal
        currentPressure = (currentPressure*9/5)+32;
    }

    // initialise the ParameterController appropriately
    ParameterController::setValue(currentPressure);
    ParameterController::updateInfo();

}


void CurrentPressureScreen::setPressureSentiment(float pressure)
{

    if (pressure >= 1000 ) { // Sunny if temperatuure is 1000mb or higher
        ParameterController::setSentiment("Sunny :-)     ");
    } else if ( pressure < 1000 && pressure >= 980 ) {
        ParameterController::setSentiment("Sunny & Clouds"); // Sunny with clouds 
    } else if ( pressure < 980 && pressure >= 960 ) {
        ParameterController::setSentiment("Cloudy        "); // Cloudy
    } else if ( pressure < 960 && pressure >= 940 ) {
        ParameterController::setSentiment("Rainy         "); // Rainy
    } else if ( pressure < 940 && pressure >= 920 ) {
        ParameterController::setSentiment("Stormy!       "); // Stormy
    } else {
        ParameterController::setSentiment("Thunderstorm!!"); // Thunderstorm
    }

}

float CurrentPressureScreen::getCurrentPressure()
{
    float pressure = getPressure(); // read temperature from BMP180 sensor
    return pressure;
}

void CurrentPressureScreen::onBack()
{
    if (debounceSuccess()) { // debouncing was successful
        playSound(); // play sound from buzzer
        nextScreen = -1;
        changeScreen = true; // user wants to view a different screen
        debounce.reset(); // reset the timer
    }


}

void CurrentPressureScreen::onEnter()
{
    if (debounceSuccess()) { // debouncing was successful
        //sound the buzzer
        playSound();
        changeUnits(); // change the units
        debounce.reset(); // reset the debounce timer
    }

}

void CurrentPressureScreen::changeUnits()
{
    // toggle the units form Cel to Fahr and back
    mbar = !mbar;

    float currentPressure = getCurrentPressure(); // get current temperature
    // update the pressure sentiment
    setPressureSentiment(currentPressure);

    if (mbar) { // units are in milibars
        ParameterController::setUnit("mbars");
    } else { // units are in Pascals
        ParameterController::setUnit("kPscl");
        //convert to kilo Pascals
        currentPressure = currentPressure/10;

    }

    // update the ParameterController appropriately
    ParameterController::setValue(currentPressure);
    ParameterController::updateInfo();

}



#endif