/**
@file BaterryStatusScreen.h


*/

#ifndef BATTERYSTATUSSCREEN_H
#define BATTERYSTATUSSCREEN_H

#include "ParameterController.h"
#include "Inputs.h"
#include "Sensors.h"

/**

@brief Displays the current battery voltage along with it's sentiment on the LCD screen \n
@brief It consists of a parameter controller\n 
@brief battery voltage is shown in Volts\n
@author Augustine Kizito K\n
@date April 2015

*/



class BatteryStatusScreen : public ParameterController // inherit ParameterController Properties
{
    
public:
    /**
    This functions manages the execution of the battery status screen
    @returns 
     -1 - Navigate to  previous screen
    */
    int start();
    /**
    Creates an instance of the battery status screen
    
    */
    BatteryStatusScreen()
        :ParameterController( "Voltage", // Parameter
                              "Volts") // Units
    {} // initialises parameter controller
    
private:

    void setVoltageSentiment( float temperature); // get temperature sentiment
    void voltageUpdate();  // update the voltage
    float getCurrentVoltage(); // get the current voltage
    void onBack(); // user pressed the back button
    void onEnter(); // user pressed the enter button

    bool changeScreen; // tracks if the user pressed a button that changes screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen
    Ticker voltTicker; // create ticker object


};

int BatteryStatusScreen::start()
{

    changeScreen = false;

    float currentVoltage = getCurrentVoltage(); // get current temperature

    backButton.mode(PullDown); // activate pullDown resistor
    enterButton.mode(PullDown); // activate pullDown resistor

    backButton.rise(this,&BatteryStatusScreen::onBack); // call onBack when the back button is pressed

    debounce.start(); // start the debouncing timer

    voltTicker.attach(this, &BatteryStatusScreen::voltageUpdate, 10); // update parameter every 10 seconds

    // initialise the ParameterController appropriately
    ParameterController::setValue(currentVoltage);
    setVoltageSentiment(currentVoltage);
    
    // Launch the Parameter Controller
    ParameterController::showInfo();

    while(changeScreen == false) { // shows the screen until the user presses the back button
        // mbed goes to sleep to save power
        Sleep();
    }

    // detach interrupts
    backButton.rise(NULL);
    voltTicker.detach();

    return nextScreen;
}

void BatteryStatusScreen::voltageUpdate()
{

    float currentVoltage = getCurrentVoltage(); // get current voltage
    // update the voltage sentiment
    setVoltageSentiment(currentVoltage);

    // initialise the ParameterController appropriately
    ParameterController::setValue(currentVoltage);
    ParameterController::updateInfo();

}


void BatteryStatusScreen::setVoltageSentiment(float batteryVoltage)
{
    if (batteryVoltage > 7.92) { // battery voltage is greater than 2.6V
        // battery has three bars
        ParameterController::setSentiment("Battery Full  ");

    } else if ( (batteryVoltage < 7.92 )&& (batteryVoltage > 6.5)) { // battery voltage is between 2.64V and 1.65V
        // battery has two bars
        ParameterController::setSentiment("Sufficient    ");

    } else if ((batteryVoltage < 6.5) && (batteryVoltage > 6)) { // battery voltage is between 1.65V and 0.66V
        // battery has one bar
        ParameterController::setSentiment("Battery Low   ");
    } else { // battery voltage is less than 0.66V
        // empty battery
        ParameterController::setSentiment("Change Battery");
    }

}

float BatteryStatusScreen::getCurrentVoltage()
{
    float voltage = getVoltage(); // read voltage
    return voltage;
}

void BatteryStatusScreen::onBack()
{
    if (debounceSuccess()) {
        playSound(); // play sound from buzzer
        nextScreen = -1;
        changeScreen = true; // user wants to view a different screen
        debounce.reset(); // reset the timer
    }


}





#endif