#ifndef RASPBERRYPISCREEN_H
#define RASPBERRYPISCREEN_H

#include "RaspiSerial.h"
Thread thread3(osPriorityNormal, (DEFAULT_STACK_SIZE * 2.25), NULL);

int piCxn = 0;      // tracks the connection of the raspberry pi
int prevpiCxn = 0;  // tracks the previous connection of the raspberry pi

class RaspberryPiScreen: public Button <RaspberryPiScreen>
{
public:
    RaspberryPiScreen(uint16_t backgroundColor);
    int start();
    void buttonPressed(int button);

private:
    void drawRaspberryPiScreen();
    void drawData();
    void updateData();
    void deAllocateMemory();
    void connectPi();
    void disconnectPi();

    // list
    ListController *list;
    // updated when a scroll event occurs
    bool scroll;
    // updated when an enter event occurs
    bool enter;
    // background color
    uint16_t bgColor;
    // text color;
    uint16_t textColor;

    int changeScreen;
    bool piConnection;  // tracks whether connection with the rpi is initiated
    bool camera;    // contains the state of the camera
    bool sensor;    // holds the state of the sensor
    bool internet;  // holds the state of the internet

};

RaspberryPiScreen::RaspberryPiScreen(uint16_t backgroundColor)
{
    bgColor = backgroundColor;
    // determine the color of text
    if (bgColor == ST7735_WHITE ) {
        textColor = ST7735_RED;
    } else if (bgColor == ST7735_BLACK) {
        textColor = ST7735_CYAN;
    }
}

int RaspberryPiScreen::start()
{
    // draw the screen
    drawRaspberryPiScreen();

    // attach interrupts to buttons
    Button<RaspberryPiScreen>::activateButtons(this);

    changeScreen = 0;
    scroll = false;
    enter = false;

    while(changeScreen==0) {

        // update the raspberyy pi data
        updateData();

        // scroll event occured
        if(scroll) {
            // perform scroll on the list
            list->scroll();
            // reset variable
            scroll = false;
        }

        // enter event occured
        if (enter) {
            // read the current option on the list
            int option = list->getCurrentOption();

            if (option == 1) {
                // make connection with pi
                //myled = 1;

                // first check connection with the pi
                //piCxn = 1;
                // pi is not yet connected
                if (piCxn == 0) {
                    // pi attempts to connect
                    piCxn = 1;
                    
                    // turn on pi LED

                    // initiate connection here
                    connectPi();

                } else {
                    // do nothing
                }


            } else {
                // disconnect with pi
                //piCxn = 3;

                // first check the pi connection
                if (piCxn == 2) { // pi connection is active
                    // pi attempts to disconnect
                    piCxn = 3;
                    disconnectPi();

                } else {
                    // do nothing
                }
                //myled = 0;
            }
            enter = false;
        }

        Thread::wait(200);
    }

    // detach interrupts to buttons
    Button<RaspberryPiScreen>::deActivateButtons();

    // free up any memory
    deAllocateMemory();

    return changeScreen;
}

void RaspberryPiScreen::updateData()
{

    // open the mail and check for messages

    
    osEvent evt = serial_ui_mail.get(0);
    if(evt.status == osEventMail) {
        serial_ui_letter *letter = (serial_ui_letter*)evt.value.p;
        piConnection = letter->piStat;
        camera = letter->camStat;
        internet = letter->cloudStat;
        sensor = letter->detectionStat;
        // delete the message
        serial_ui_mail.free(letter);
    }
    
    // if the pi is in transition 
    if (piCxn == 1 && piConnection == true)
    {
        piCxn = 2;    
    }
    if (piCxn == 3 && piConnection == false)
    {
        piCxn = 0;    
    }
    
    




    // update the title
    if(piCxn == prevpiCxn) {
        // do nothing
    } else {


        tft.setTextColor(bgColor);
        tft.setCursor((tft.width()/2)-30,4);
        // first delete the previous connection words
        switch(prevpiCxn) {

                // delete "DISCONNECTED"
            case 0:
                tft.printf("DISCONNECTED");
                break;
                // delete "CONNECTING..."
            case 1:
                tft.printf("CONNECTING...");
                break;
                // delete "CONNECTED"
            case 2:
                tft.printf("CONNECTED");
                break;
            case 3:
                // delete "DISCONNECTING..."
                tft.printf("DISCONNECTING...");
                break;

            default:
                break;
        }

        tft.setTextColor(textColor);
        tft.setCursor((tft.width()/2)-30,4);
        // secondly write the new connection words
        switch(piCxn) {

                // write "DISCONNECTED"
            case 0:
                tft.printf("DISCONNECTED");
                // turn off PI LED
                RPiLED = 0;
                break;
                // write "CONNECTING..."
            case 1:
                tft.printf("CONNECTING...");
                // turn on PI LED
                RPiLED = 1;
                break;
                // write "CONNECTED"
            case 2:
                tft.printf("CONNECTED");
                break;
            case 3:
                // write "DISCONNECTING..."
                tft.printf("DISCONNECTING...");
                break;

            default:
                break;
        }

    }



    // update the prevCxnStatus
    prevpiCxn = piCxn;

}

void RaspberryPiScreen::drawRaspberryPiScreen()
{
    // draw static data here
    tft.setTextColor(textColor);

    // display title
    tft.setCursor((tft.width()/2)-50,4);
    tft.printf("PI:");

    // display camera
    tft.setCursor(0,20);
    tft.printf("camera:");

    // display cloud
    tft.setCursor(0,35);
    tft.printf("internet:");

    // display sensor
    tft.setCursor(0,50);
    tft.printf("sensor:");



    // construct the list
    list = new ListController( 0,
                               63,
                               "Turn Pi on",
                               "Yes",
                               "No",
                               "",
                               "",
                               bgColor,
                               2);

    // draw the list
    list->drawList();

    // initialise dynamic data
    tft.setTextColor(textColor);
    tft.setCursor((tft.width()/2)-30,4);
    // first delete the previous connection words
    switch(prevpiCxn) {

            // delete "DISCONNECTED"
        case 0:
            tft.printf("DISCONNECTED");
            break;
            // delete "CONNECTING..."
        case 1:
            tft.printf("CONNECTING...");
            break;
            // delete "CONNECTED"
        case 2:
            tft.printf("CONNECTED");
            break;
        case 3:
            // delete "DISCONNECTING..."
            tft.printf("DISCONNECTING...");
            break;

        default:
            break;
    }
}

void RaspberryPiScreen::deAllocateMemory()
{
    // deallocate memory that was created dynamically
    delete list;
}

void RaspberryPiScreen::buttonPressed(int button)
{
    switch(button) {

        case BACKBUTTON:
            // navigate to the previous screen
            changeScreen = -1;
            break;

        case SCROLLBUTTON:
            // scroll to the next option
            scroll = true;
            break;

        case ENTERBUTTON:
            // navigate to the selected option
            enter = true;
            break;

        default:
            // do nothing
            break;
    }
}

void RaspberryPiScreen::connectPi()
{
    ui_serial_letter *letter2 = ui_serial_mail.alloc();
    letter2->activateSerial = true;
    ui_serial_mail.put(letter2);
    
    // start the thread 
     // launch serial communication with raspberry pi
    
    thread3.start(raspiSerial);
}

void RaspberryPiScreen::disconnectPi()
{
    ui_serial_letter *letter2 = ui_serial_mail.alloc();
    letter2->activateSerial = false;
    ui_serial_mail.put(letter2);
}

#endif