#ifndef MAINMENUSCREEN_H
#define MAINMENUSCREEN_H

#include "Icons.h"
#include "Inputs.h"

class MainMenuScreen : public Button <MainMenuScreen>
{
public:
    MainMenuScreen(uint16_t backgroundColor);
    int start();
    void buttonPressed(int button);

private:
    void drawMainMenuScreen();
    void initialiseIcons();
    void drawIcons();
    void drawText();
    void updateIcons();
    void shiftBoundary();
    void deAllocateMemory();

    // colors
    uint16_t textColor;     // color for text
    uint16_t bgColor;       // background color
    uint16_t byColor;        // boundary color


    int boundaryShifter;    // tracks the location of the boundary
    int changeScreen;
    bool scroll;    // updated when a scroll event occurs

    // Icons
    BatteryIcon *batteryIcon;
    OutputIcon *outputIcon;
    SolarIcon *solarIcon;
    SettingsIcon *settingsIcon;

};


MainMenuScreen::MainMenuScreen(uint16_t backgroundColor)
{
    // constuctor function



    bgColor = backgroundColor;

    // set text color
    // determine the color of text
    if (bgColor == ST7735_WHITE ) {
        textColor = ST7735_BLACK;
        byColor = ST7735_CYAN;
    } else if (bgColor == ST7735_BLACK) {
        textColor = ST7735_WHITE;
        byColor = ST7735_CYAN;
    }

}

int MainMenuScreen::start()
{
    
    drawMainMenuScreen();   
    
    // attach interrupts to buttons
    Button<MainMenuScreen>::activateButtons(this);
    
    changeScreen = 0;
    
    while (changeScreen == 0){
        updateIcons();
        
        // scroll event occured
        if(scroll)
        {
            shiftBoundary();
            scroll = false;    
        }
        
        //wait_ms(500);
        Thread::wait(200);       
    }
    
    // remember to detach buttons
    Button<MainMenuScreen>::deActivateButtons();
    
    // remember to deallocate memory
    deAllocateMemory();
    
    return changeScreen;
}

void MainMenuScreen::drawMainMenuScreen()
{
    // initialise icons
    initialiseIcons();
    
    // draw the screens icons first
    drawIcons();

    // draw the text under the icons
    drawText();

    // Highlight the first icon
    boundaryShifter = 1;
    shiftBoundary();
}

void MainMenuScreen::initialiseIcons()
{
    // Battery Icon
    // X Coord
    int batXCoord = 15;
    // Y Coord
    int batYCoord = 5;
    // percentage
    int batteryPercentage = getBatteryPercentage();
    // batteryStatus
    int batteryStatus = getBatteryStatus();
    bool batteryCharging = false;
    if (batteryStatus > 0 )
    {
        batteryCharging = true;    
    }
    
    // Constructor
    batteryIcon = new BatteryIcon(batXCoord, batYCoord,bgColor,batteryPercentage,batteryCharging);

    // Output Icon Coordinates
    // X Coord
    int outXCoord = 100;
    // Y Coord
    int outYCoord = 8;
    // Constructor
    outputIcon = new OutputIcon(outXCoord, outYCoord, bgColor);

    // Solar Icon Coordinates
    // X Coord
    int solXCoord = 5;
    // Y Coord
    int solYCoord = 63;
    // Constructor
    solarIcon = new SolarIcon(solXCoord, solYCoord, bgColor);

    // Settings Icon Coordinates
    // X Coord
    int setXCoord = 100;
    // Y Coord
    int setYCoord = 66;
    // Constructor
    settingsIcon = new SettingsIcon(setXCoord, setYCoord, bgColor);
}

void MainMenuScreen::drawIcons()
{
    // draw the battery icon
    batteryIcon->drawBatteryIcon();

    // draw solar icon
    solarIcon->drawSolarIcon();

    // draw output icon
    outputIcon->drawOutputIcon();

    // draw settings icon
    settingsIcon->drawSettingsIcon();
}

void MainMenuScreen::drawText()
{
    // text under the icons

    // Batteries
    tft.setCursor(7, 50);
    tft.setTextColor(textColor);
    tft.setTextWrap(true);
    tft.printf("Batteries");

    // Output
    tft.setCursor(110, 50);
    tft.setTextColor(textColor);
    tft.setTextWrap(true);
    tft.printf("Output");

    // Solar
    tft.setCursor(17, 117);
    tft.setTextColor(textColor);
    tft.setTextWrap(true);
    tft.printf("Solar");

    // Settings
    tft.setCursor(103, 117);
    tft.setTextColor(textColor);
    tft.setTextWrap(true);
    tft.printf("Settings");
}

void MainMenuScreen::shiftBoundary()
{
    // selects the next icon to have a boundary

    if (boundaryShifter == 1) {

        // delete settings
        tft.drawRect(96,114,61,13,bgColor);
        tft.drawRect(97,115,59,11,bgColor);

        // draw the battery boundary
        tft.drawRect(0,47,70,13,byColor);
        tft.drawRect(1,48,68,11,byColor);

    } else if (boundaryShifter == 2) {

        // delete battery
        tft.drawRect(0,47,70,13,bgColor);
        tft.drawRect(1,48,68,11,bgColor);

        // draw output
        tft.drawRect(103,47,50,13,byColor);
        tft.drawRect(104,48,48,11,byColor);

    } else if (boundaryShifter == 3) {

        // delete output
        tft.drawRect(103,47,50,13,bgColor);
        tft.drawRect(104,48,48,11,bgColor);


        // draw solar
        tft.drawRect(10,114,42,13,byColor);
        tft.drawRect(11,115,40,11,byColor);

    } else if (boundaryShifter == 4) {

        // delete solar
        tft.drawRect(10,114,42,13,bgColor);
        tft.drawRect(11,115,40,11,bgColor);


        // draw settings
        tft.drawRect(96,114,61,13,byColor);
        tft.drawRect(97,115,59,11,byColor);

    }
    
    // ready the next icon to have boundary
    boundaryShifter++;

    // ensure the boundary tracker remains in a loop
    if (boundaryShifter > 4) {
        boundaryShifter = 1;
    }

}

void MainMenuScreen::updateIcons()
{
    // update the icons
    
    // get the overall percentage of the battery
    int percentage = getBatteryPercentage();
    
    // get the charge status of the battery
    bool batteryCharging = getBatteryStatus();
    
    // get the output status
    bool outputStatus = getOutputStatus();
    
    // get the radiation status of the solar panel
    bool radiationReceived = getSolarStatus();

    // set the battery percentage accordingly
    batteryIcon->setBatteryPercentage(percentage,batteryCharging);
    
    // animate the output icon accordingly
    outputIcon->animateOutputIcon(outputStatus);

    // animate the solar icon accordingly
    solarIcon->animateSolarIcon(radiationReceived);
}
void MainMenuScreen::deAllocateMemory()
{
    // free up memory that was dynamically created
    delete batteryIcon;
    delete outputIcon;
    delete solarIcon;
    delete settingsIcon;    
}

void MainMenuScreen::buttonPressed(int button)
{
    switch(button){
        
        case BACKBUTTON:
            // navigate to the previous screen
            //changeScreen = -1;
            break;
            
        case SCROLLBUTTON:
            // move boundary to the next icon
            //shiftBoundary();
            scroll = true;
            break;
        
        case ENTERBUTTON:
            // navigate to the selected option
            if(boundaryShifter == 1){
                changeScreen = 4;    
            } else {
                changeScreen = boundaryShifter-1;   
            }
        default:
            // do nothing
            break;    
    }
}

#endif