#ifndef UTILITYSCREEN_H
#define UTILITYSCREEN_H


DigitalOut utilityPin(p8);
int uCxn = 0;      // tracks the connection of the raspberry pi
int prevuCxn = 0;  // tracks the previous connection of the raspberry pi

class UtilityScreen: public Button <UtilityScreen>
{
public:
    UtilityScreen(uint16_t backgroundColor);
    int start();
    void buttonPressed(int button);

private:
    void drawUtilityScreen();
    void drawData();
    void displayStaticData();
    void displayDynamicData();
    void updateData();
    void deAllocateMemory();
    void turnOnUtilityPort();
    void turnOffUtilityPort();

    // list
    ListController *list;
    // updated when a scroll event occurs
    bool scroll;
    // updated when an enter event occurs
    bool enter;
    // background color
    uint16_t bgColor;
    // text color;
    uint16_t textColor;
    
    float current;
    float voltage;
    float power;

    int changeScreen;

};

UtilityScreen::UtilityScreen(uint16_t backgroundColor)
{
    bgColor = backgroundColor;
    // determine the color of text
    if (bgColor == ST7735_WHITE ) {
        textColor = ST7735_RED;
    } else if (bgColor == ST7735_BLACK) {
        textColor = ST7735_CYAN;
    }
}

int UtilityScreen::start()
{
    // initial values
    current = -1.0;
    voltage = -1.0;
    power = -1.0;
    
    // draw the screen
    drawUtilityScreen();

    // attach interrupts to buttons
    Button<UtilityScreen>::activateButtons(this);

    changeScreen = 0;
    scroll = false;
    enter = false;

    while(changeScreen==0) {

        // update the raspberyy pi data
        updateData();

        // scroll event occured
        if(scroll) {
            // perform scroll on the list
            list->scroll();
            // reset variable
            scroll = false;
        }

        // enter event occured
        if (enter) {
            // read the current option on the list
            int option = list->getCurrentOption();

            if (option == 1) {
                // make connection with pi
                //myled = 1;

                // first status of utility port
                //piCxn = 1;
                
                // utility port is not yet turned on
                if (uCxn == 0) {
                    // turn on utlity port
                    uCxn = 1;

                    // initiate connection here
                    turnOnUtilityPort();

                } else {
                    // do nothing
                }


            } else {
                // disconnect with pi
                //piCxn = 3;

                // first check the pi connection
                if (uCxn == 1) { // utility port is turned on
                    // turn off the utility port
                    uCxn = 0;
                    turnOffUtilityPort();

                } else {
                    // do nothing
                }
                //myled = 0;
            }
            enter = false;
        }

        Thread::wait(200);
    }

    // detach interrupts to buttons
    Button<UtilityScreen>::deActivateButtons();

    // free up any memory
    deAllocateMemory();

    return changeScreen;
}

void UtilityScreen::updateData()
{

    displayDynamicData();
    
   

}

void UtilityScreen::drawUtilityScreen()
{
    // draw static data here
    tft.setTextColor(textColor);
    

    displayStaticData();



    // construct the list
    list = new ListController( 0,
                               63,
                               "Turn Port on",
                               "Yes",
                               "No",
                               "",
                               "",
                               bgColor,
                               2);

    // draw the list
    list->drawList();
    

}

void UtilityScreen::displayStaticData()
{ 
    
    // the static data is displayed here e.g headings
    tft.setTextWrap(true);
    tft.setTextColor(textColor);

    // display title
    char title[20];
    if (uCxn){
     strcpy(title,"UTILITY PORT ON");   
    } else {
     strcpy(title,"UTILITY PORT OFF");  
    }
    tft.setCursor((tft.width()/2)-50,4);
    tft.printf("%s",title);

    // display voltage
    tft.setCursor(0, 20);
    tft.printf("VOLTAGE[V]:");

    // display minimum
    tft.setCursor(0, 35);
    tft.printf("CURRENT[mA]:");

    // display capacity remaining
    tft.setCursor(0, 50);
    tft.printf("POWER[W]:");

}


void UtilityScreen::displayDynamicData()
{
    // first retrieve data from the sensor suite
    SensorSuite suite;
    ConsumptionModel model = suite.getConsumptionData();
    
    float newCurrent;
    float newVoltage;
    float newPower;
    
    newCurrent = model.consumptionCurrent;
    newVoltage = model.consumptionVoltage;
    newPower = newCurrent*newVoltage*0.001;
    
    
    

    tft.setTextWrap(true);
    //tft.setTextColor(textColor);
    
    // the dynamic data is displayed here e.g values
    
    if (newVoltage == voltage)
    {
        // do nothing    
    }  else {
        
        tft.setCursor(tft.width()/2, 20);
        // first delete the previous value
        tft.setTextColor(backgroundColor);
        tft.printf("%+6.3f",voltage);
        
        tft.setCursor(tft.width()/2, 20);
        // then write the new voltage values
        tft.setTextColor(textColor);
        tft.printf("%+6.3f",newVoltage);
    }
    
    if (newCurrent == current){
        // do nothing    
    } else {
        
        // first delete the previous value
        tft.setCursor(tft.width()/2, 35);
        tft.setTextColor(backgroundColor);
        tft.printf("%+6.3f",current);
        
        // then write the new current values
        tft.setCursor(tft.width()/2, 35);
        tft.setTextColor(textColor);
        tft.printf("%+6.3f",newCurrent);
    }
    
     if (newPower == power)
    {
        // do nothing    
    }  else {
        
        tft.setCursor(tft.width()/2, 50);
        // first delete the previous value
        tft.setTextColor(backgroundColor);
        tft.printf("%+6.3f",power);
        
        tft.setCursor(tft.width()/2, 50);
        // then write the new voltage values
        tft.setTextColor(textColor);
        tft.printf("%+6.3f",newPower);
    }
    

   
    
    voltage = newVoltage;
    current = newCurrent;
    power = newPower;
}


void UtilityScreen::deAllocateMemory()
{
    // deallocate memory that was created dynamically
    delete list;
}

void UtilityScreen::buttonPressed(int button)
{
    switch(button) {

        case BACKBUTTON:
            // navigate to the previous screen
            changeScreen = -1;
            break;

        case SCROLLBUTTON:
            // scroll to the next option
            scroll = true;
            break;

        case ENTERBUTTON:
            // navigate to the selected option
            enter = true;
            break;

        default:
            // do nothing
            break;
    }
}

void UtilityScreen::turnOnUtilityPort()
{
    /**
    ui_serial_letter *letter2 = ui_serial_mail.alloc();
    letter2->activateSerial = true;
    ui_serial_mail.put(letter2);
    
    // start the thread 
     // launch serial communication with raspberry pi
    
    thread3.start(raspiSerial);
    
    **/
    
    // turn on utility port
    utilityPin = 1;
    utilityLED = 1;
    
    // delete text "UTILITY PORT OFF"
    tft.setTextColor(backgroundColor);
    tft.setCursor((tft.width()/2)-50,4);
    tft.printf("UTILITY PORT OFF");
    
    
    // print text "UTILITY PORT ON"
    tft.setTextColor(textColor);
    tft.setCursor((tft.width()/2)-50,4);
    tft.printf("UTILITY PORT ON");
}

void UtilityScreen::turnOffUtilityPort()
{
    /**
    ui_serial_letter *letter2 = ui_serial_mail.alloc();
    letter2->activateSerial = false;
    ui_serial_mail.put(letter2);
    **/
    
    // turn off utility port
    utilityPin = 0;
    utilityLED = 0;
    
    // delete text "UTILITY PORT ON"
    tft.setTextColor(backgroundColor);
    tft.setCursor((tft.width()/2)-50,4);
    tft.printf("UTILITY PORT ON");
    
    
    // print text "UTILITY PORT OFF"
    tft.setTextColor(textColor);
    tft.setCursor((tft.width()/2)-50,4);
    tft.printf("UTILITY PORT OFF");
}

#endif