#ifndef LOGRECORD_H
#define LOGRECORD_H

#include "mbed.h"

// logging parameters
#define SOLAR_POWER 1
#define POWER_CONSUMPTION 2

// durations
#define THIRTY_SECONDS 1
#define ONE_MINUTE 2

class LogRecord
{
public:
    LogRecord();
    LogRecord( char iD[]);
    void initialiseRecord(MbedJSONValue &temp, int paramter, int duration);
    void initialiseRecord(MbedJSONValue &temp, int ID);
    bool updateRecord(MbedJSONValue &temp); // returns true if update successful, false if completed
    int getParameter(); // get the current parameter
    int getID();    // get the ID of the record
    char *getName();    // get the name of the parameter
    float getMaximum(); // get the maximum value
    float getMinimum(); // get the minimum value
    float getAverage(); // get the average value
    int getStatus();  // get the status
    int getTimeStamp();    // get the time of recording
    int getDuration();  // get the duration of the log
    float getPercentage();  // get the percentage time of the log completed
    
private:
    int parameter;
    int status;
    char timeleft[35];
    float maximum;
    float minimum;
    float average;
    float percentage;
    char name[15];
    char logID[35];
    int id;
    int count;  // the number of times the record is updated
    int totalCount; // the total number of data points to be measured
    int viewed; // checks if the log has been viewed
    int timeStamp;   // time the log was taken
    int duration;   // the time it took
};

LogRecord::LogRecord()
{
    // set the name to be empty
    strcpy(name,"");
}

LogRecord::LogRecord( char iD[])
{
    // copy iD string into logID
    strcpy(logID,iD);
}

void LogRecord::initialiseRecord(MbedJSONValue &temp, int ID)
{
    // copy all the data into the record
    
    // id used to determine its location
    id = ID;
    
    timeStamp = temp[logID]["t"].get<int>();
    
    // parameter
    parameter = temp[logID]["p"].get<int>();
    
    if(parameter == SOLAR_POWER)
    {
        strcpy(name,"solar");    
    }else if(parameter == POWER_CONSUMPTION){
        strcpy(name,"consumption");    
    }
    
    // maximum
    int maximumInt = temp[logID]["mx"].get<int>();
    maximum = maximumInt/100;
    
    // minimum
    int minimumInt = temp[logID]["mn"].get<int>();
    minimum = minimumInt/100;
    
    // average
    int averageInt = temp[logID]["av"].get<int>();
    average = averageInt/100;
    
    // status
    status = temp[logID]["s"].get<int>();
    // duration
    duration = temp[logID]["d"].get<int>();
    // percentage 
    int percentageInt = temp[logID]["%"].get<int>();
    percentage = percentageInt/10;
    // viewed
    viewed = temp[logID]["v"].get<int>();
    
    

}

void LogRecord::initialiseRecord(MbedJSONValue &temp, int param, int durn)
{
    // set counter to zero
    count = 0;
    // set the parameter
    temp[logID]["p"] = param;
    // also determine which value/sensor to be stored

    // set the duration
    temp[logID]["d"] = durn;
    // determine the total data points
    switch(durn) {
            // the interrupt that logs data is called every 10s
        case THIRTY_SECONDS:
            totalCount = 3;
            break;
        case ONE_MINUTE:
            totalCount = 6;
            break;
        default:
            totalCount = 0;
            break;
    }

    // set the time logging started
    temp[logID]["t"] = static_cast<int>(time(NULL));;

    // set the status to logging
    temp[logID]["s"] = 1;

    // set default max
    maximum = 0.0;
    // set default min
    minimum = 0.0;
    // set default average
    average = 0.0;
}

bool LogRecord::updateRecord(MbedJSONValue &temp)
{
    // increment counter
    count++;

    if (count > totalCount) {
        // change status to complete
        temp[logID]["s"] = 2;
        return false;   // logging complete
    }

    // value used to store the read value
    float value = 30;

    // calculate maximum
    if( value > maximum)
        maximum = value;
    // calculate minimum
    if( value < minimum)
        minimum = value;
    // calclute average
    average = (average + value )/ count;
    // calculate %age complete
    float percentage = count/totalCount;

    // write data to JSON Object
    temp[logID]["mx"] = (int)(maximum*100);
    temp[logID]["mn"] = (int)(minimum*100);
    temp[logID]["av"] = (int)(average*100);
    temp[logID]["%"] = (int)(percentage*100);

    return true;
}

int LogRecord::getParameter()
{
    return parameter;    
}

char *LogRecord::getName()
{
    return name;    
}

int LogRecord::getID()
{
    return id;    
}

float LogRecord::getMaximum()
{
    return maximum;    
}

float LogRecord::getMinimum()
{
    return minimum;    
}

float LogRecord::getAverage()
{
    return average;    
}

int LogRecord::getStatus()
{
    return status;    
}

int LogRecord::getTimeStamp()
{
    return timeStamp;    
}

int LogRecord::getDuration()
{
    return duration;    
}

float LogRecord::getPercentage()
{
    return percentage;
}
#endif