#ifndef GRAPHCONTROLLER_H
#define GRAPHCONTROLLER_H

class GraphController
{

public:
    GraphController(uint16_t backgroundColor, char title[]);
    void plotData();                           // plots the data on the graph
    void drawAxes();                           // draws the axes on the map

private:
    int pointNumber;          // the total number of X Coordintes
    int originWidth;           // where the x axis starts
    int originHeight;         // where the y axis starts
    int previousValue;         // previous x coordinate
    int position;
    int counter;
    uint16_t bgColor;
    uint16_t textColor;
    uint16_t plotColor;
    char t[32];

    int valStore[130];              // stores all avalialble x Cordinates i.e before plotting
    int oldValStore[130];           // stores outdated x Coordinates i.e after plotting

    void drawLeftToRight();                    // plots data from left to right
    void drawRightToLeft();                    // plots data from right to left
    void printXAxis(int direction);            // prints labels on the X Axis depending on direction
    void increment(int position, int val);     // plots data when before splace on the x axis runs out
    void shift(int val);                       // intenlligently plots more data on the graph by moving plots to the left

};

GraphController::GraphController(uint16_t backgroundColor, char title[])
{
    pointNumber = 130;          // the total number of X Coordintes
    originWidth = 15;           // where the x axis starts
    originHeight = 116;         // where the y axis starts

    strcpy(t,title);

    bgColor = backgroundColor;

    if (bgColor == ST7735_BLACK) {
        textColor = ST7735_WHITE;
        plotColor = ST7735_YELLOW;
    } else if (bgColor == ST7735_WHITE) {
        textColor = ST7735_BLACK;
        plotColor = ST7735_BLUE;
    }
}

void GraphController::drawAxes()
{
    tft.setTextColor(textColor);

    // first draw the title
    tft.setCursor(25,0);
    tft.printf("%s",t);

    // draw the axes
    tft.drawFastHLine(originWidth-1-3,originHeight,pointNumber+10,textColor);
    tft.drawFastVLine(originWidth-1,originHeight+1-100,100+3,textColor);

    // draw the demarcations
    // y axis
    tft.drawFastHLine(originWidth-1-3,originHeight-30,3,textColor);
    tft.drawFastHLine(originWidth-1-3,originHeight-60,3,textColor);
    tft.drawFastHLine(originWidth-1-3,originHeight-90,3,textColor);

    // x axis
    tft.drawFastVLine(originWidth-1+26,originHeight+1,3,textColor);
    tft.drawFastVLine(originWidth-1+(26*2),originHeight+1,3,textColor);
    tft.drawFastVLine(originWidth-1+(26*3),originHeight+1,3,textColor);
    tft.drawFastVLine(originWidth-1+(26*4),originHeight+1,3,textColor);
    tft.drawFastVLine(originWidth-1+(26*5),originHeight+1,3,textColor);


    // label the X Axis left to right direction
    printXAxis(0);


    // change the rotation
    tft.setCursor(originWidth-10,originHeight-3);
    tft.printf("0");

    tft.setCursor(originWidth-10,originHeight-2-30);
    tft.printf("3");

    tft.setCursor(originWidth-10,originHeight-2-(30*2));
    tft.printf("6");

    tft.setCursor(originWidth-10,originHeight-2-(30*3));
    tft.printf("9");
    
    position = 0;
    counter = 0;
    previousValue = -1;         // previous x coordinate
}

void GraphController::printXAxis(int direction)
{
    // direction of zero means left to right

    // direction one means right to left

    // first set text color to deleting color
    tft.setTextColor(bgColor);

    if (direction == 0) {

        // first clear all the previous right to left numbers
        drawRightToLeft();

        // draw numbers form left to right
        tft.setTextColor(textColor);
        drawLeftToRight();

    } else if(direction == 1) {

        // first clear all the previous left to right numbers
        drawLeftToRight();

        // draw numbers from left to right
        tft.setTextColor(textColor);
        drawRightToLeft();
        tft.setTextColor(plotColor);

    }

    // reset to the drawing color

}

void GraphController::drawLeftToRight()
{
    tft.setCursor(originWidth-3,originHeight+5);
    tft.printf("0");

    tft.setCursor(originWidth-8+(25),originHeight+5);
    tft.printf("2.5");

    tft.setCursor(originWidth-1+(25*2),originHeight+5);
    tft.printf("5");

    tft.setCursor(originWidth-8+(25*3),originHeight+5);
    tft.printf("7.5");

    tft.setCursor(originWidth-3+(25*4),originHeight+5);
    tft.printf("10");

    tft.setCursor(originWidth-8+(25*5),originHeight+5);
    tft.printf("12.5");
}

void GraphController::drawRightToLeft()
{

    tft.setCursor(originWidth-15,originHeight+5);
    tft.printf("12.5");

    tft.setCursor(originWidth-5+(25),originHeight+5);
    tft.printf("10");

    tft.setCursor(originWidth-8+(25*2),originHeight+5);
    tft.printf("7.5");

    tft.setCursor(originWidth+(25*3),originHeight+5);
    tft.printf("5");

    tft.setCursor(originWidth-5+(25*4),originHeight+5);
    tft.printf("2.5");

    tft.setCursor(originWidth+2+(25*5),originHeight+5);
    tft.printf("0");
}

void GraphController::plotData()
{
    //static int position = 0;
    //static int counter = 0;
    // first check how many much space is left
    //numStore[position-1]= position;
    int val = getSolarPower();

    // read the val

    //counter makes sure the value is only displayed
    // only once every 5 times the graph is updated

    if (counter == 0) {
        if (val != previousValue) {
            // first delete the previous value
            tft.setCursor(2,9);
            tft.setTextColor(bgColor);
            tft.printf("%dW",previousValue);

            // then write the new value
            tft.setCursor(2,9);
            tft.setTextColor(plotColor);
            tft.printf("%dW",val);
        } else {
            // do nothing
        }

        previousValue = val;

    }


    // esure the counter is reset accordingly i.e every 5 times
    counter++;
    if (counter > 5) {
        counter = 0;
    }


    // if the space on the X Axis is enough
    if (position < pointNumber) {
        // update the array with the new value
        valStore[position] = val;

        // plot the value in the available position
        //increment(position, val);
        tft.drawFastVLine(originWidth+(position*1),originHeight-val,val,plotColor);

        // plot in the following position the next time
        position++;
    } else {

        // intelligently plot the new value on the graph
        shift(val);
    }

    // ensure change of X Axis direction only occurs once
    if (position==pointNumber) {
        printXAxis(1);
        position++;
    }
}

void GraphController::increment(int position, int val)
{

}

void GraphController::shift(int val)
{
    // all coordinates in the graph are shifted to the left
    // therefore creating space for one more coordinate to be plotted


    // first, store previous values in the apprpriate array array
    for(int i = 0; i < pointNumber; i++) {
        oldValStore[i] = valStore[i];
    }


    // second, shift all the values in the array to the left

    int bufferVal = -1;

    for (int i = (pointNumber-1); i >= 0; i--) {

        int currentSpot = i;
        int nextSpot = i-1;

        int nextVal = 0;

        if(bufferVal != -1) {
            nextVal = bufferVal;
        } else {
            nextVal = valStore[currentSpot];
        }

        bufferVal =  valStore[nextSpot];
        valStore[nextSpot] = nextVal;
    }

    valStore[pointNumber-1] = val;

    // third, intelligently plot the data onto the graph
    int deltaVal;
    for (int i = 0; i < pointNumber; i++) {

        // first calculate difference between the new value and the old value
        deltaVal = valStore[i] - oldValStore[i];

        // if new value is bigger the old value
        if (deltaVal > 0) {
            // only add the extra height required
            tft.drawFastVLine(originWidth+(i*1),originHeight-oldValStore[i]-deltaVal,deltaVal,plotColor);
        }
        // if new value is smaller than the old value
        else if(deltaVal < 0) {
            // only remmove the excess height
            tft.drawFastVLine(originWidth+(i*1),originHeight-oldValStore[i],(deltaVal*-1),bgColor);
        }
        // if the values are the same
        else {
            // do nothing
        }
    }
}


#endif