#ifndef SOLARVALUESCREEN_H
#define SOLARVALUESCREEN_H

class SolarValueScreen: Button <SolarValueScreen>
{
public:
    SolarValueScreen(uint16_t);
    int start();
    void buttonPressed( int button );
private:
    float voltage;
    float current;
    float power;
    int changeScreen;
    uint16_t backgroundColor;   // stores the background color
    uint16_t textColor;         // stores the text color
    void displayStaticData();   // displays data that does not change
    void displayDynamicData();  // displays data that changes
};

SolarValueScreen::SolarValueScreen(uint16_t bgColor)
{

    backgroundColor = bgColor;
    // determine the color of text
    if (bgColor == ST7735_WHITE ) {
        textColor = ST7735_BLACK;
    } else if (bgColor == ST7735_BLACK) {
        textColor = ST7735_WHITE;
    } else if (bgColor == ST7735_BLUE) {
        textColor = ST7735_WHITE;
    }
}

int SolarValueScreen::start()
{
    // initial values
    current = -1.0;
    voltage =  -1.0;
    power = -1.0;
    
    // Populate the Screen
    displayStaticData();
    displayDynamicData();

    changeScreen = 0;
    // attach interrupts to buttons
    Button<SolarValueScreen>::activateButtons(this);

    while(changeScreen == 0) {
        displayDynamicData();
        //wait_ms(500);
        Thread::wait(200);   
    }

    // dettach interrupts to buttons
    Button<SolarValueScreen>::deActivateButtons();

    return changeScreen;
}

void SolarValueScreen::displayStaticData()
{
    // the static data is displayed here e.g headings
    tft.setTextWrap(true);
    tft.setTextColor(textColor);

    // display title
    tft.setCursor((tft.width()/2)-50,4);
    tft.printf("SOLAR PANEL");

    // display voltage
    tft.setCursor(0, 20);
    tft.printf("VOLTAGE(V)");

    // display current
    tft.setCursor(0, 50);
    tft.printf("CURRENT(mA)");

    // display capacity remaining
    tft.setCursor(0, 80);
    tft.printf("POWER(W)");
}

void SolarValueScreen::displayDynamicData()
{
    // first retrieve data from the sensor suite
    SensorSuite suite;
    SolarModel model = suite.getSolarData();
    
    float newCurrent;
    float newVoltage;
    float newPower;
    
    // the dynamic data is displayed here e.g values
    int added = 10;
    tft.setTextWrap(true);
    tft.setTextColor(textColor);
    
    newCurrent = model.solarCurrent;
    newVoltage = model.solarVoltage;
    newPower = newCurrent*newVoltage*0.001;

    // display the voltage
    if (newVoltage == voltage)
    {
        // do nothing    
    }  else {
        
        tft.setCursor(0, 20+added);
        // first delete the previous value
        tft.setTextColor(backgroundColor);
        tft.printf("%+6.3f",voltage);
        
        tft.setCursor(0, 20+added);
        // then write the new voltage values
        tft.setTextColor(textColor);
        tft.printf("%+6.3f",newVoltage);
    }
    
    // display the current
    if (newCurrent == current){
        // do nothing    
    } else {
        
        // first delete the previous value
        tft.setCursor(0, 50+added);
        tft.setTextColor(backgroundColor);
        tft.printf("%+6.3f",current);
        
        // then write the new current values
        tft.setCursor(0, 50+added);
        tft.setTextColor(textColor);
        tft.printf("%+6.3f",newCurrent);
    }

    // display capacity remaining
    if(newPower == power){
        // do nothing
    } else {
        // first delete the previous value
        tft.setCursor(0, 80+added);
        tft.setTextColor(backgroundColor);
        tft.printf("%+6.3f",power);
        
        // then write the new current values
        tft.setCursor(0, 80+added);
        tft.setTextColor(textColor);
        tft.printf("%+6.3f",newPower);
        
    }
    
    voltage = newVoltage;
    current = newCurrent;
    power = newPower;

}

void SolarValueScreen::buttonPressed( int button )
{
    switch(button) {

        case BACKBUTTON:
            // navigate to the previous screen
            changeScreen = -1;
            break;

        case SCROLLBUTTON:
            // scroll to the next option
            //list->scroll();
            break;

        case ENTERBUTTON:
            // navigate to the selected option
            //changeScreen = ListController::getCurrentOption();
            //showDetails = true;
            break;

        default:
            // do nothing
            break;
    }
}
#endif