#ifndef BATTERYSCREEN_H
#define BATTERYSCREEN_H

#define BATTERY_ONE 1
#define BATTERY_TWO 2

#include "SensorSuite.h"
#include "SensorModel.h"

class BatteryScreen: public Button <BatteryScreen>
{
public:
    BatteryScreen(uint16_t);    // constructor
    int start( int batt );
    void buttonPressed( int button );
private:
    char current_text[20];
    char voltage_text[20];
    float current;
    float voltage;
    float power;
    int changeScreen;
    int battery;
    uint16_t backgroundColor;   // stores the background color
    uint16_t textColor;
    
    void retireveBatteryData();
    void displayStaticData();   // the information that doesnt change
    void displayDynamicData();  // the information that changes

};

BatteryScreen::BatteryScreen(uint16_t bgColor)
{
    backgroundColor = bgColor;
    // determine the color of text
    if (bgColor == ST7735_WHITE ) {
        textColor = ST7735_BLACK;
    } else if (bgColor == ST7735_BLACK) {
        textColor = ST7735_WHITE;
    } else if (bgColor == ST7735_BLUE) {
        textColor = ST7735_WHITE;
    }

}

int BatteryScreen::start( int batt )
{
    // initial values
    current = -1.0;
    voltage = -1.0;
    power = -1.0;
    
    battery = batt; // determine the battery
    
    // Populate the Screen
    displayStaticData();
    displayDynamicData();

    changeScreen = 0;
    // attach interrupts to buttons
    Button<BatteryScreen>::activateButtons(this);

    while(changeScreen == 0) {
        displayDynamicData();
        //wait_ms(500);
        Thread::wait(200);   
    }

    // dettach interrupts to buttons
    Button<BatteryScreen>::deActivateButtons();

    return changeScreen;
}

void BatteryScreen::displayStaticData()
{
    
    
    
    
    
    // the static data is displayed here e.g headings
    tft.setTextWrap(true);
    tft.setTextColor(textColor);

    // display title
    char title[20];
    switch(battery) {
        case BATTERY_ONE:
            strcpy(title,"BATTERY ONE");
            break;
        case BATTERY_TWO:
            strcpy(title,"BATTERY TWO");
            break;
        default:
            break;
    }
    tft.setCursor((tft.width()/2)-50,4);
    tft.printf("%s",title);

    // display voltage
    tft.setCursor(0, 20);
    tft.printf("VOLTAGE[V]");

    // display minimum
    tft.setCursor(0, 50);
    tft.printf("CURRENT[mA]");

    // display capacity remaining
    tft.setCursor(0, 80);
    tft.printf("POWER[W]");

}

void BatteryScreen::displayDynamicData()
{
    // first retrieve data from the sensor suite
    SensorSuite suite;
    BatteryModel model = suite.getBatteryData();
    
    float newCurrent;
    float newVoltage;
    float newPower;
    
    switch(battery) {
        case BATTERY_ONE:
            newCurrent = model.batteryOneCurrent;
            newVoltage = model.batteryOneVoltage;
            break;
        case BATTERY_TWO:
            newCurrent = model.batteryTwoCurrent;
            newVoltage = model.batteryTwoVoltage;
            break;
        default:
            break;
    }
    
    newPower = newCurrent*newVoltage*0.001;
    
    int added = 10;
    tft.setTextWrap(true);
    //tft.setTextColor(textColor);
    
    // the dynamic data is displayed here e.g values
    
    if (newVoltage == voltage)
    {
        // do nothing    
    }  else {
        
        tft.setCursor(0, 20+added);
        // first delete the previous value
        tft.setTextColor(backgroundColor);
        tft.printf("%+6.3f",voltage);
        
        tft.setCursor(0, 20+added);
        // then write the new voltage values
        tft.setTextColor(textColor);
        tft.printf("%+6.3f",newVoltage);
    }
    
    if (newCurrent == current){
        // do nothing    
    } else {
        
        // first delete the previous value
        tft.setCursor(0, 50+added);
        tft.setTextColor(backgroundColor);
        tft.printf("%+6.3f",current);
        
        // then write the new current values
        tft.setCursor(0, 50+added);
        tft.setTextColor(textColor);
        tft.printf("%+6.3f",newCurrent);
    }
    
    // display the power
    if (newPower == power){
        // do nothing    
    } else {
        
        // first delete the previous value
        tft.setCursor(0, 80+added);
        tft.setTextColor(backgroundColor);
        tft.printf("%+6.3f",power);
        
        // then write the new current values
        tft.setCursor(0, 80+added);
        tft.setTextColor(textColor);
        tft.printf("%+6.3f",newPower);
    }
    

    // display capacity remaining
    //tft.setCursor(0, 80+added);
    
    voltage = newVoltage;
    current = newCurrent;
    power = newPower;
}

void BatteryScreen::buttonPressed( int button )
{
    switch(button) {

        case BACKBUTTON:
            // navigate to the previous screen
            changeScreen = -1;
            break;

        case SCROLLBUTTON:
            // scroll to the next option
            //list->scroll();
            break;

        case ENTERBUTTON:
            // navigate to the selected option
            //changeScreen = ListController::getCurrentOption();
            //showDetails = true;
            break;

        default:
            // do nothing
            break;
    }
}


#endif