#ifndef BATTERIESSCREEN_H
#define BATTERIESSCREEN_H

#include "ListController.h"
#include "SensorSuite.h"
#include "SensorModel.h"

class BatteriesScreen : public Button <BatteriesScreen>
{
public:
    BatteriesScreen(uint16_t backgroundColor);
    int start();
    void buttonPressed(int button);

private:

    void drawBatteriesScreen();
    void drawTile();
    void updateTile();
    void deAllocateMemory();

    // Icons
    BatteryIcon *batteryOne;
    BatteryIcon *batteryTwo;

    // list
    ListController *list;

    // updated when a scroll event occurs
    bool scroll;
    // updated when an enter event occurs
    bool enter;
    // background color
    uint16_t bgColor;

    int changeScreen;
};

BatteriesScreen::BatteriesScreen(uint16_t backgroundColor)
{

    bgColor = backgroundColor;
}

int BatteriesScreen::start()
{



    // draw the screen
    drawBatteriesScreen();

    // attach interrupts to buttons
    Button<BatteriesScreen>::activateButtons(this);

    changeScreen = 0;
    scroll = false;
    enter = false;


    while (changeScreen == 0) {

        // update the title accordingly
        updateTile();

        // scroll event occured
        if (scroll) {
            // perform scroll on the list
            list->scroll();
            // reset variable
            scroll = false;
        }

        // enter event occured
        if (enter) {
            // read the current option on the list
            changeScreen = list->getCurrentOption();
        }
        //wait_ms(200);
        Thread::wait(200);
    }

    // dettach interrupts to buttons
    Button<BatteriesScreen>::deActivateButtons();

    // free up some memory
    deAllocateMemory();

    return changeScreen;
}

void BatteriesScreen::updateTile()
{

    // percentage
    int batteryOnePercentage = getBatteryOnePercentage();
    int batteryTwoPercentage = getBatteryTwoPercentage();

    // batteryStatus
    int batteryStatus = getBatteryStatus();

    // track state of individual battery
    bool batteryOneCharging = false;
    bool batteryTwoCharging = false;

    if (batteryStatus == 1 || batteryStatus == 3) {
        batteryOneCharging = true;
    }
    if (batteryStatus == 2 || batteryStatus == 3) {
        batteryTwoCharging = true;
    }

    batteryOne->setBatteryPercentage(batteryOnePercentage, batteryOneCharging);
    batteryTwo->setBatteryPercentage(batteryTwoPercentage, batteryTwoCharging);
}


void BatteriesScreen::drawBatteriesScreen()
{

    // draw the tile
    drawTile();

    // construct the list
    list = new ListController( 0,
                               63,
                               "Batteries",
                               "Battery One",
                               "Battery Two",
                               "",
                               "",
                               bgColor,
                               2);

    // draw the list
    list->drawList();

}

void BatteriesScreen::drawTile()
{
    // first draw the blue background
    tft.fillRect(0,0,tft.width(),63,ST7735_BLUE);

    // retrieve batteryPercentage
    int batteryPercentage = getBatteryPercentage();
    
    // batteryStatus
    int batteryStatus = getBatteryStatus();

    // track state of individual battery
    bool batteryOneCharging = false;
    bool batteryTwoCharging = false;

    if (batteryStatus == 1 || batteryStatus == 3) {
        batteryOneCharging = true;
    }
    if (batteryStatus == 2 || batteryStatus == 3) {
        batteryTwoCharging = true;
    }

    // Coordinates for battery one
    int x = 5;
    int y = 5;

    // initialise the first battery icon
    batteryOne = new BatteryIcon(x,y,ST7735_BLUE,batteryPercentage,batteryOneCharging);

    // Coordinates for battery two
    int x1 = 83;
    int y1 = 5;

    // Initialise the second battery icon
    batteryTwo = new BatteryIcon(x1,y1,ST7735_BLUE,batteryPercentage,batteryTwoCharging);

    // draw the batteryIcons
    batteryOne->drawBatteryIcon();
    batteryTwo->drawBatteryIcon();

    // draw underlying text
    tft.setCursor(13,50);
    tft.setTextColor(ST7735_WHITE);
    tft.printf("B1");

    tft.setCursor(91,50);
    tft.setTextColor(ST7735_WHITE);
    tft.printf("B2");

}

void BatteriesScreen::deAllocateMemory()
{
    // deallocate memory that was created dynamically
    delete batteryOne;
    delete batteryTwo;
    delete list;

}

void BatteriesScreen::buttonPressed(int button)
{
    switch(button) {

        case BACKBUTTON:
            // navigate to the previous screen
            changeScreen = -1;
            break;

        case SCROLLBUTTON:
            // scroll to the next option
            scroll = true;
            break;

        case ENTERBUTTON:
            // navigate to the selected option
            enter = true;
            break;

        default:
            // do nothing
            break;
    }
}


#endif

