#ifndef SENSORSUITE_H
#define SENSORSUITE_H

#include "INA219.h"
#include "SensorModel.h"
I2C i2c(p9,p10);

/**

This class is going to manage everything to do with sensors
It should be able to receive the sensor data from the current sensors
And then perform some calculatations where necessary
This data is aftewards placed in mailboxes where it can be accessed by
other threads.

**/


bool alternate = false;

class SensorSuite
{
public:
    SensorSuite();  // constructor
    void begin();   // launches the sensor suite thread

    // functions used to inteface with suite
    BatteryModel getBatteryData();   // retrieve information about batteries
    SolarModel getSolarData();       // retrieve information about solar panel
    ConsumptionModel getConsumptionData();  // retrieve information about consumption


private:
    void setupSensors(); // this ensures the sensors are ready to collect data
    void readSensorData();  // reads all the data from all sensors
    void batteryPercentageEstimate();   // estimates the battery percentage
    void suiteOperation();  // function that operated indefinitely

    // voltage-current sensors
    static INA219 *sensorOne;      // sensor 1
    static INA219 *sensorTwo;      // sensor 2
    static INA219 *sensorThree;    // sensor 3
    static INA219 *sensorFour;     // sensor 4

    // parameters
    static float b1Current;    // current from battery one
    static float b1Voltage;    // voltage from battery one
    static float b2Current;    // current from battery one
    static float b2Voltage;    // voltage from battery one
    static float spCurrent;    // current form solar panel
    static float spVoltage;    // voltage from solar panel
    static float outVoltage;   // voltage at output
    static float outCurrent;   // current at output

    static bool batteriesSwitched;    // determines which battery is which after switching

    // use mutexes - restrict access to sensor data
    static Mutex battery_mutex; //  restricts access to battery data
    static Mutex solar_mutex;   // restricts access to solar data
    static Mutex consumption_mutex;  // restricts access to consumption data
    static Mutex general_mutex;      // restrics general acess to data during aquisition




};

SensorSuite::SensorSuite()
{
    // determine what mailbox to use: should probably be a pointer
}

void SensorSuite::begin()
{
    // intialise the current sensors
    setupSensors();

    // start infinite thread to get data
    //Thread thread(osPriorityNormal, (DEFAULT_STACK_SIZE * 2.25), NULL);
    //thread.start(callback(this,&SensorSuite::suiteOperation));
    suiteOperation();

    // maybe start infinite thread to receive events

}

void SensorSuite::setupSensors()
{
    // dynamically configure the sensors
    sensorOne = new INA219(i2c, INA219_ADDR_GV);    // for battery one
    sensorTwo = new INA219(i2c, INA219_ADDR_VG);    // for battery two
    sensorThree = new INA219(i2c, INA219_ADDR_GG);  // for solar panel
    sensorFour = new INA219(i2c, INA219_ADDR_VV);  // for power consumption

    // querry the position of the batteries
}

void SensorSuite::suiteOperation()
{
    // this is the infinitely running thread

    while(true) {

        // restrict access to battery data
        battery_mutex.lock();

        // read values from the batteries
        // switching circuit is in default position
        if(!batteriesSwitched) {

            // read values from the first battery
            b1Current = sensorOne->read_current();
            b1Voltage =  sensorOne->read_bus_voltage();

            // read values from the second battery
            b2Current = sensorTwo->read_current();
            b2Voltage = sensorTwo->read_bus_voltage();

        } else { // switch is in alternate position
            
            // read values from the first battery
            b1Current = sensorTwo->read_current();
            b1Voltage = sensorTwo-> read_bus_voltage();

            // read values from the second battery
            b2Current = sensorOne->read_current();
            b2Current = sensorOne->read_bus_voltage();
        }
        
        if (b1Voltage < 3.4)
        {
            // turn on LED
            batteryOneLowLED = 1;
            
        } else {
            // turn off LED
            batteryOneLowLED = 0;
        }
        
        if (b2Voltage < 3.4)
        {
            // turn on LED    
            batteryTwoLowLED = 1;
        } else {
            
            // turn off LED    
            batteryTwoLowLED = 0;
        }
        // release access to battery data
        battery_mutex.unlock();

        // restrict access to solar data
        solar_mutex.lock();
        
        // read values from the solar panel
        spCurrent = sensorThree->read_current();
        spVoltage = sensorThree->read_bus_voltage();
        
        // release access to solar data
        solar_mutex.unlock();


        // restrict access to consumption data
        consumption_mutex.lock();

        // read values from the output sensor
        outVoltage = sensorFour->read_bus_voltage();
        outCurrent = sensorFour->read_current();
        
        // release access to consumption data
        consumption_mutex.unlock();

        // unlock the general mutex
        //general_mutex.unlock();

        Thread::wait(50);   // wait ms

    }

    // read the data points

    // load the datapoints into the mailboxes


}

BatteryModel SensorSuite::getBatteryData()
{
    // restrict access to battery data to a single thread
    battery_mutex.lock();

    // collect data about the batteries
    BatteryModel model(b1Current,b1Voltage,b2Current,b2Voltage);

    // unlock access to battery data to different threads
    battery_mutex.unlock();
    
    return model;
}

SolarModel SensorSuite::getSolarData()
{
    // restrict access to solar data to a single thread
    solar_mutex.lock();

    // collect solar data
    SolarModel model(spVoltage,spCurrent);

    // allow access to solar data to different threads
    solar_mutex.unlock();

    return model;

}

ConsumptionModel SensorSuite::getConsumptionData()
{
    // restrict access to consumption data to a single thread
    consumption_mutex.lock();
    
    // collect consumption data
    ConsumptionModel model(outVoltage, outCurrent);
    
    // allow access to consumption data to different threads
    consumption_mutex.unlock();   
    
    return model; 
}

// ensure that sensor data is initialised
// voltage-current sensors
INA219* SensorSuite::sensorOne = NULL;      // sensor 1
INA219* SensorSuite::sensorTwo = NULL;      // sensor 2
INA219* SensorSuite::sensorThree = NULL;    // sensor 3
INA219* SensorSuite::sensorFour = NULL;     // sensor 4

// parameters
float SensorSuite::b1Current = 0.0;    //current from battery one
float SensorSuite::b1Voltage = 0.0;    // voltage from battery one
float SensorSuite::b2Current = 0.0;    // current from battery one
float SensorSuite::b2Voltage = 0.0;    // voltage from battery one
float SensorSuite::spCurrent = 0.0;    // current form solar panel
float SensorSuite::spVoltage = 0.0;    // voltage from solar panel
float SensorSuite::outVoltage = 0.0;   // voltage at output
float SensorSuite::outCurrent = 0.0;   // current at output

bool SensorSuite::batteriesSwitched = false;    // determines which battery is which after switching

// use mutexes - restrict access to sensor data
Mutex SensorSuite::battery_mutex; //  restricts access to battery data
Mutex SensorSuite::solar_mutex;   // restricts access to solar data
Mutex SensorSuite::consumption_mutex;  // restricts access to consumption data
Mutex SensorSuite::general_mutex;      // restrics general acess to data during aquisition





#endif