#ifndef OUTPUTSCREEN_H
#define OUTPUTSCREEN_H

#include "ListController.h"

class OutputScreen:public Button <OutputScreen>
{
public:
    OutputScreen(uint16_t backgroundColor);
    int start();
    void buttonPressed(int button);

private:
    void drawOutputScreen();
    void drawTile();
    void updateTile();
    void drawWattage(float wattage, uint16_t color);

    int changeScreen;
    float previousWattage;
    ListController *list;   // manages the list
    bool scroll;    // updated when a scroll event occurs
    bool enter;     // updated when an enter event occurs
    uint16_t bgColor;   // background color

};

OutputScreen::OutputScreen(uint16_t backgroundColor)
{

    bgColor = backgroundColor;
}

int OutputScreen::start()
{
    previousWattage = -1;

    // draw the screen
    drawOutputScreen();

    // attach interrupts to buttons
    Button<OutputScreen>::activateButtons(this);

    changeScreen = 0;
    scroll = false;
    enter = false;


    while (changeScreen == 0) {
        updateTile();

        // scroll event occured
        if (scroll) {
            // perform scroll on the list
            list->scroll();
            // reset variable
            scroll = false;
        }

        // enter event occured
        if (enter) {
            // read the current option on the list
            changeScreen = list->getCurrentOption();
        }
        //wait_ms(500);
        Thread::wait(200);
    }

    // release the memory held by the list
    delete list;

    // dettach interrupts to buttons
    Button<OutputScreen>::deActivateButtons();

    return changeScreen;
}


void OutputScreen::drawOutputScreen()
{

    // draw the tile
    drawTile();

    // construct the list
    list = new ListController( 0,
                               63,
                               "Output",
                               "Raspberry Pi",
                               "Utility Port",
                               "",
                               "",
                               backgroundColor,
                               2);


    // draw the list
    list->drawList();
    //Button<BatteriesScreen>::activateButtons();
}



void OutputScreen::drawTile()
{   
    // draw the green background
    tft.fillRect(0,0,tft.width(),63,ST7735_YELLOW);
    
    // draw the text
    tft.setTextSize(1);
    tft.setCursor(5,5);
    tft.setTextColor(ST7735_BLUE);
    tft.printf("Power:");
    
    // update the power values
    updateTile();
}

void OutputScreen::updateTile()
{
    // read the current power
    float wattage = getConsumptionPower();
    
    // temporary storage
    char wattageString[5];
    
    // tracks previously measured power
    // ensure first value is ridiculous 
    // to enable print on startup
    //static float previousWattage = -1;
    
    //  convert float to char*
    //  to round off to 1 decimal place
    sprintf(wattageString,"%.1f",wattage);    
    
    // convert back to float
    wattage = atof(wattageString);
    
    // multiply by 10
    wattage = wattage*10;
    
    // convert float to integer
    float currentWattage = wattage;
    
    if(currentWattage != previousWattage){
        
        // first delete the previous wattage
        drawWattage(previousWattage,ST7735_YELLOW);
        // then write the new wattage
        drawWattage(currentWattage,ST7735_RED);
        
    
    }else{
        // do nothing    
    }
    
    // update the previous wattage
    previousWattage = currentWattage;
}

void OutputScreen::drawWattage(float wattage, uint16_t color)
{
    // convert float to integer
    int wattageInt = wattage;
    
    // divide by 10
    int firstInt = wattageInt/10;
    
    // print the quotient as the first value
    int secondInt = wattageInt%10;
    
    // the coordinates of the first large digit
    int startWidth = 40;
    int startHeight = 15;
    
    // The first Large Digit
    tft.setTextSize(6);
    tft.setCursor(startWidth, startHeight);
    tft.setTextColor(color);
    tft.printf("%d",firstInt);
    
    // the decimal point
    tft.setCursor(startWidth+ 28, startHeight + 21);
    tft.setTextSize(3);
    tft.printf(".");
    
    // The second Large Digit
    tft.setCursor(startWidth + 45, startHeight);
    tft.setTextSize(6);
    tft.printf("%d", secondInt);
    
    // The power units
    tft.setCursor(startWidth + 80, startHeight + 15);
    tft.setTextSize(3);
    tft.printf("W");
    
    // reset the text size to smallest  
    tft.setTextSize(1);  
}

void OutputScreen::buttonPressed(int button)
{
    switch(button) {

        case BACKBUTTON:
            // navigate to the previous screen
            changeScreen = -1;
            break;

        case SCROLLBUTTON:
            // scroll to the next option
            scroll = true;
            break;

        case ENTERBUTTON:
            // navigate to the selected option
            //changeScreen = ListController::getCurrentOption();
            enter = true;
            break;

        default:
            // do nothing
            break;
    }
}

#endif