#ifndef BATTERYICON_H
#define BATTERYICON_H

#include "mbed.h"

class BatteryIcon
{
public:
    //BatteryIcon();
    BatteryIcon(uint8_t xCoordinate, uint8_t yCoordinate, uint16_t backgroundColor, int batteryPercentage, bool chargeStatus);
    void setBatteryPercentage(int batteryPercentage, bool chargeStatus);
    void drawBatteryIcon();
    void removeBatteryIcon();

private:
    uint8_t xCoord;
    uint8_t yCoord;
    uint16_t bgColor;
    int previousPercentage;
    int percentage;
    int batteryLevel;
    int previousBatteryLevel;
    int previousBatteryColor;
    bool chargeStat;
    bool thunderboltShowing;
    bool battInitialised;
    uint16_t deleteColor;
    uint16_t textColor;


    uint16_t determineBatteryColor();
    void drawBatteryOutline();
    void displayPercentage(int pc, uint16_t tc);
    void fillBattery();
    void drawThunderbolt();
    void removeThunderbolt();

};

/**
BatteryIcon::BatteryIcon(){
    
}
**/

BatteryIcon::BatteryIcon(uint8_t xCoordinate, uint8_t yCoordinate, uint16_t backgroundColor, int batteryPercentage, bool chargeStatus)
                            : previousPercentage(-1),previousBatteryLevel(0), previousBatteryColor(0),thunderboltShowing(false)
{
    // make all necessary initialisations
    xCoord = xCoordinate;
    yCoord = yCoordinate;
    bgColor = backgroundColor;
    chargeStat = chargeStatus;
    percentage = batteryPercentage;

    // determine the color to delete stuff
    deleteColor = bgColor;

    // determine the color of text
    if (bgColor == ST7735_WHITE ) {
        textColor = ST7735_BLACK;
    } else if (bgColor == ST7735_BLACK) {
        textColor = ST7735_WHITE;
    } else if (bgColor == ST7735_BLUE) {
        textColor = ST7735_WHITE;    
    }
}

void BatteryIcon::drawBatteryIcon()
{

    // draw the outline first
    drawBatteryOutline();

    // dispaly the battery Percentage
    setBatteryPercentage(percentage,chargeStat);

}

void BatteryIcon::removeBatteryIcon()
{

}

void BatteryIcon::setBatteryPercentage(int percentage, bool chargeStatus)
{

    // calculate batter level;
    float batteryLevelFloat = (percentage*32.0)/100;
    batteryLevel = batteryLevelFloat;

    // write the battery percentage
    if (previousPercentage == -1) {

        // write a new percentage
        displayPercentage(percentage, textColor);

    } else if (percentage == previousPercentage) {
        // do nothing
    } else {

        // first delete the current percentage
        displayPercentage(previousPercentage, bgColor);

        // write a new percentage
        displayPercentage(percentage, textColor);
    }
    
    

    // fill the battery
    fillBattery();
    
    

    // draw Thunderbolt at the end

    // first check if battery is charging
    if (chargeStatus) {

        // check if thunderbolt is not displayed
        if (!thunderboltShowing) {
            // draw the thundebolt
            drawThunderbolt();
        } else {
            // do nothing
        }
    } else { // battery is not charging

        // check if thunderbolt is showing
        if (thunderboltShowing) {
            // remove the thunderbolt
            removeThunderbolt();
        } else {
            // do nothing
        }

    }
    
    // record the previous data
    previousPercentage = percentage;
    previousBatteryLevel = batteryLevel;
    
    

}

void BatteryIcon::displayPercentage(int pc,uint16_t tc )
{
    // dispalay the battery percentage
    tft.setCursor(xCoord+28, yCoord+20);
    tft.setTextColor(tc);
    tft.setTextSize(2);
    tft.printf("%ld%%",pc);
    tft.setTextSize(1);
}

void BatteryIcon::drawThunderbolt()
{

    int16_t height = yCoord;
    int16_t width = xCoord + 40;

    // draw first diagonal lines
    for (int n = 0; n < 3; n++ ) {
        tft.drawLine(width+n+4, height+1,width+n+1,height+7,ST7735_YELLOW);
    }
    // draw horizontal lines
    for (int n = 0; n < 2; n++) {
        tft.drawFastHLine(width+3,height+6+n,4,ST7735_YELLOW);
    }
    // draw second horizontal lines
    for (int n = 0; n < 3; n++ ) {
        tft.drawLine(width+n+6, height+6,width+3+n, height+11,ST7735_YELLOW);
    }
    // draw filled triangle
    tft.fillTriangle(width, height+12,width+8, height+12,width+4, height+16,ST7735_YELLOW);
    
    thunderboltShowing = true;
}

void BatteryIcon::removeThunderbolt()
{
    
    // here the thunderbolt lines are replaced by the background color, hence
    // making the thunderbolt disappear
    
    int16_t height = yCoord;
    int16_t width = xCoord + 40;

    // draw first diagonal lines
    for (int n = 0; n < 3; n++ ) {
        tft.drawLine(width+n+4, height+1,width+n+1,height+7,bgColor);
    }
    // draw horizontal lines
    for (int n = 0; n < 2; n++) {
        tft.drawFastHLine(width+3,height+6+n,4,bgColor);
    }
    // draw second horizontal lines
    for (int n = 0; n < 3; n++ ) {
        tft.drawLine(width+n+6, height+6,width+3+n, height+11,bgColor);
    }
    // draw filled triangle
    tft.fillTriangle(width, height+12,width+8, height+12,width+4, height+16,bgColor);
    
    thunderboltShowing = false;
}

void BatteryIcon::fillBattery()
{

    if ( batteryLevel > previousBatteryLevel ) {

        // first determine the color of lines
        uint16_t theColor = determineBatteryColor();

        // increase the number of lines
        for (int y = previousBatteryLevel; y < batteryLevel; y++) {
            tft.drawFastHLine(xCoord + 1, yCoord + 38-y,23,theColor);
        }
    } else if (batteryLevel < previousBatteryLevel) {

        // decrease the number of lines
        for (int y = previousBatteryLevel; y > batteryLevel; y--) {
            tft.drawFastHLine(xCoord + 1, yCoord + 38-y,23,bgColor);
        }

        // first determine the color of lines
        uint16_t theColor = determineBatteryColor();

    } else {
        // do nothing
    }

}

uint16_t BatteryIcon::determineBatteryColor()
{

    uint16_t color;

    if (batteryLevel >= 21) {

        //first check if its already green
        if (previousBatteryColor != 1) {

            // make the colors green
            for (int y = 0; y <= batteryLevel; y++) {
                tft.drawFastHLine(xCoord + 1,yCoord + 38-y,23, ST7735_GREEN);
            }

            previousBatteryColor = 1;
        }

        // update the new battery color
        color = ST7735_GREEN;

    } else if (batteryLevel < 21 && batteryLevel >= 7 ) {

        //first check if its already yellow
        if (previousBatteryColor != 2) {

            // make the colors yellow
            for (int y = 0; y <= batteryLevel; y++) {
                tft.drawFastHLine(xCoord + 1,yCoord + 38-y,23, ST7735_YELLOW);
            }

            previousBatteryColor = 2;
        }

        // update the new battery color
        color = ST7735_YELLOW;
    } else {
        //first check if its already red
        if (previousBatteryColor != 3) {

            // make the colors red
            for (int y = 0; y <= batteryLevel; y++) {
                tft.drawFastHLine(xCoord + 1,yCoord + 38-y,23, ST7735_RED);
            }


            previousBatteryColor = 3;
        }

        // update the new battery color
        color = ST7735_RED;
    }

    return color;

}

void BatteryIcon::drawBatteryOutline()
{
    // small rectangle
    tft.drawRect( xCoord + 8, yCoord,10,6,ST7735_GREEN);
    
    // big rectangle
    tft.drawRect( xCoord , yCoord + 5,25,35,ST7735_GREEN);
}

#endif