#include <mbed.h>
#include <math.h>
#include "L298_H.h"
#include "IRSensor_H.h"
#include "ColourSensor_H.h"


/*
Team Design Project 3 Group 1
Uploaded code V2.2

-----PINOUT-----------------------

Motor Left Control :
    - Motor Enable PWM :     PTA12
    - Backwards Dout :       PTA4
    - Forward Dout :         PTA5

Motor Right Control :
    - Motor Enable PWM :     PTA13
    - Backwards Dout :       PTD5
    - Forward Dout :         PTD0
    
IR Sensor Pinout :
    - Left Sensor :         PTC6
    - Middle Left Sensor :  PTC5
    - Middle Sensor :       PTC4
    - Middle Right Sensor : PTC3
    - Right Sensor :        PTC0
    
Colour Sensor Pinout :
    - Colour Sensor 1:      PTB10
    - Colour Sensor 2:      PTB11
    - Solenoid :            PTE20

UltraSonic Sensor Pinout :
    - Trigger Pin 1 :       PTE2
    - Echo pin 1 :          PTE3
    - Trigger Pin 2         PTE5
    - echopin2 :            PTE4

Serial Sensor Pinout :
    - Esp Tx/Rx :           PTE22, PTE23
    - PC Tx/Rx :            USBTX, USBRX
    - psGroung :            PTC16
    - redLed :              PTC13
    - blueLed :             PTC12
    
Switch and Debug :
    - Motor On :            PTE21         
    - myled :               LED_RED

-----CODE BRIEF----------------------------
The code puts together the class and code from the L298, IRsensor and ColourSensor headers to
enable a the TDP3 Group1 robot to move, follow a line, detect a colour, detect an obstacle and
communicate in Bluetooth.
The code presents three main tickers which sample data from sensors from the Line Follower, 
Colour Detection and Obstacle Detection. Further Timeouts and Interrupt are implemented.
  
*/



//SERIAL Pinout===
RawSerial ser(PTE22, PTE23); //TX, RX
Serial pc(USBTX, USBRX);
DigitalOut psGround(PTC16);
DigitalOut redLed(PTC13);
DigitalOut blueLed(PTC12);

//========================

//ULTRASONIC Pinout====
DigitalOut trigpin(PTE2);
DigitalIn echopin(PTE3);
DigitalOut trigpin2(PTE5);
DigitalIn echopin2(PTE4);
//======================

//Switch and Debug pinout=======
DigitalIn motorOn(PTE21);       //switch which turns on or off the motors  
DigitalOut myled(LED_RED);
//==============================

//IR init=======================
Ticker IRsampling;              //ticker for the IR sampling
//================================

//COLOURSENSOR INIT======
//initialize the Colour sensor object
ColourSensor ColourSensor(PTB10, PTB11, PTE20);
Timeout enableColour;
Ticker detectColour;
Timeout reactivateColour;
Timeout solenoidBackOn;
Timeout CheckColour;

void ColourCheck();
//=========================

//US initialisation ===========
Ticker US_Tic;
Timer US_Timer;      //Timer used to record the time duration
bool US_flag = false;  
bool US_stop = false;   

int getDis(DigitalIn echopin, DigitalOut trigPin);
void US_inter();
//==============================

//SERIAL init===================
char temp;
bool serflag = false;
bool enBle = false;
bool serStop = false;
char gain;
float Kp = 0.0, Ki = 0.0, Kd = 0.0, noErSpeed = 0.0, turnSpeed = 0.0;

void ReadRpi();
void ReadSer(L298* controlLeft, L298* controlRight);
//==============================

void enableColourDet() {
    detectColour.attach(callback(&ColourSensor, &ColourSensor::readSensor), 0.05);   // Set call rate of colour detection function
    
}





//MAIN FUNCTION==================================================================
int main(){
    


//US +-------------
float distance = 0;
float distance2 = 0;
//-----------------

//SER +---------------------
ser.format(8, SerialBase::None, 1);                 //Serial frame of 8 bit, no parity bit and 1 stop bit
ser.baud(9600);                                     //Serial baudrate set at 9600
psGround = false;
redLed = false;
blueLed = false;
//--------------------------

bool toggleIR;



//initialize the IRsensor object, the different parameters can be tuned at will (Kp,Ki,Kd,noErSpeed,turnSpeed)
IRSensor IRSensor(PTC6, PTC5, PTC4, PTC3, PTC0, 0.9, 0, 2.3 ,0.16,0.1);//0.58,0,2.6,0.16,0.08 || 0.65, 0.0 , 3.0, 0.15, 0.11

//initialise the class twice, one controls the left part and the other the right
L298 controlLeft(PTA13,PTD5,PTD0);
L298 controlRight(PTA12,PTA4,PTA5);


//Interrupt Routines======================
enableColour.attach(&enableColourDet, 1.0);
IRsampling.attach(callback(&IRSensor, &IRSensor::Sample), 0.008);
ser.attach(&ReadRpi,Serial::RxIrq);
US_Tic.attach(&US_inter,0.3);
//========================================


//Initialise the variables changed by the phone
Kp = IRSensor.m_Kp; 
Ki = IRSensor.m_Ki;
Kd = IRSensor.m_Kd;
noErSpeed = IRSensor.m_noErSpeed;
turnSpeed = IRSensor.m_turnSpeed;

//motor init
controlLeft.SetSpeed(0);
controlRight.SetSpeed(0);

myled = 1;
gain = 0;


while(1){
    
    //enable and disable the IR process when in bluetooth mode
    if(!enBle ||!serStop){
    if(motorOn && !US_stop){
        
        //IR sensors==========================================================================
        if(toggleIR != IRSensor.m_toggle){
        //waits for an interrupt to occur, then performs all the calculations
            
            IRSensor.WeightPID();
            IRSensor.CalculatePID();
            IRSensor.MotorControl();
        
            if(IRSensor.m_dirL != IRSensor.m_prevDirL) controlLeft.SetDirection(IRSensor.m_dirL);
            if(IRSensor.m_dirR != IRSensor.m_prevDirR) controlRight.SetDirection(IRSensor.m_dirR);
    
            controlLeft.SetSpeed(fabs(IRSensor.m_speedL));
            controlRight.SetSpeed(fabs(IRSensor.m_speedR));
    
            IRSensor.m_prevDirL = IRSensor.m_dirL;
            IRSensor.m_prevDirR = IRSensor.m_dirR;
        
        
            //printf("rightSpeed : %f, leftSpeed : %f \r", IRSensor.m_speedL, IRSensor.m_speedR);    
            toggleIR = IRSensor.m_toggle;
            }
        
        //=========================================================================================
        }
    }
    
    else if(!motorOn || US_stop || serStop){
        //turns off the the motors
        controlLeft.Stop();
        controlRight.Stop();    
        } 
    
        
        
    
    //Serial Communication =============================================================================
    if(serflag == 1){
    //read serial and assign Kp values        
        
        ReadSer(&controlLeft, &controlRight);
        
        //make the phone values equal to the one used by the PID
        IRSensor.m_Kp = Kp;
        IRSensor.m_Ki = Ki;
        IRSensor.m_Kd = Kd;
        IRSensor.m_noErSpeed = noErSpeed;
        IRSensor.m_turnSpeed = turnSpeed;
        
        serflag = 0;
        myled = !myled;   
        }
    //===================================================================================================
    
    
    //COLOUR SENSOR======================================================================================
        
    if (ColourSensor.toggleConst && ColourSensor.toggleA) {
        // Call convert to convert sensor logic
        ColourSensor.convert();
        
        if(!ColourSensor.diskHave){
            blueLed = 0;
            redLed = 0;}
            
        if(ColourSensor.newDetection){
             //New Detection found, detach the interrupt, trigger a timeout to check the colour
             detectColour.detach();
             CheckColour.attach(ColourCheck, 0.5); 
             }
        ColourSensor.toggleA = 0;
    }
        
    if (ColourSensor.toggleConst && ColourSensor.toggleB && ColourSensor.toggleC) {
        // Process colour sensor data
        ColourSensor.process();
            
        //Convert the colour of the disk to the correction the robot has to make
        if(ColourSensor.diskColour && ColourSensor.diskHave){
                IRSensor.m_color = -5;
                blueLed = 1;
                redLed = 0;
                }
        else if(!ColourSensor.diskColour && ColourSensor.diskHave){
                IRSensor.m_color = 5;
                blueLed = 0;
                redLed = 1;
                }
        
        reactivateColour.attach(callback(&ColourSensor, &ColourSensor::makeColourActive), 2.0);
            
        ColourSensor.toggleB = 0;
    }
    //==========================================================================================
       
    
    //Ultrasonic Sensor =============================================================================
    if(US_flag){
        //disable irq to ensure correct readings
        //get distance of both sensors and stop the robot if and object is at less than 20cm
        __disable_irq(); 
    
        distance = 0;
        distance = getDis(echopin, trigpin);
    
        distance2 = 0;
        distance = getDis(echopin2, trigpin2);
    
        __enable_irq(); 
        
        //check distance
        if (((distance < 20.0) && (distance > 0)) || ((distance2 < 20.0) && (distance2 > 0)) ) { 
            US_stop = 1;
            myled = 1;
            }
        else {
            US_stop = 0;
            myled = 0;
            }
        US_flag = 0;
        
        }
       
    }
}






void ReadRpi(){
//attached to interrupt
//reads and store the char sent on Rx in temp
    
    if(ser.readable()) temp = ser.getc();
    serflag =  1;
}



void ReadSer(L298* controlLeft, L298* controlRight){
    //reads the information in temp 
    //if temp is either p,i or d, it assigns the value to gain
    //else if temp is + or -, it either increases or decreases the chosen parameter
        
    if(temp == 'p' || temp == 'i' || temp == 'd' || temp == 's' || temp == 'e' || temp == 'x' || temp == 'z' || temp == 'b' || temp == 'n'){
        gain = temp;
        }
    
    switch(gain){
        case 'p': 
            if(temp == '+' )Kp += 0.01;                  
            else if(temp == '-' && Kp > 0) Kp -= 0.01;     
            break;
        case 'i':
            if(temp == '+') Ki += 0.001;
            else if(temp == '-' && Ki > 0) Ki -= 0.0001;    
            break;
        case 'd':
            if(temp == '+') Kd += 0.1;   
            else if(temp == '-' && Kd > 0) Kd -= 0.1;    
            break;
        case 's':
            if(temp == '+') noErSpeed += 0.1;   
            else if(temp == '-' && noErSpeed > 0) noErSpeed -= 0.1; 
            break;
        case 'v':
            if(temp == '+') turnSpeed += 0.1;   
            else if(temp == '-' && turnSpeed > 0) turnSpeed -= 0.1; 
            break; 
        case 'x': //stop
            serStop = true;
            break;
        case 'z': 
            serStop = false;
            break;
        case 'b':
            enBle = true;
            break;
        case 'n':
            enBle = false;
            break;
                    
        default:
            break;
            }
        
    
    if (enBle == true){
        //by inputing b, the robot will become controlled by the phone user
        //inputting a direction will apply this direction for until a new direction is given
        
        switch(temp){
            case'8': //forward
            
                controlLeft->SetDirection(1);
                controlRight->SetDirection(1);
                controlLeft->SetSpeed(0.2);
                controlRight->SetSpeed(0.2);
                
                break; 
            case'4':  //left
            
                controlLeft->SetDirection(0);
                controlRight->SetDirection(1);
                controlLeft->SetSpeed(0.2);
                controlRight->SetSpeed(0.2);
                
                break; 
            case'2': //backwards
                
                controlLeft->SetDirection(0);
                controlRight->SetDirection(0);
                controlLeft->SetSpeed(0.2);
                controlRight->SetSpeed(0.2);
            
                break; 
            case'6': //right
            
                controlLeft->SetDirection(1);
                controlRight->SetDirection(0);
                controlLeft->SetSpeed(0.2);
                controlRight->SetSpeed(0.2);
                
                break; 
            default: break;
            
        }
    }
        
    serflag = 0;
}



int getDis (DigitalIn echopin, DigitalOut trigPin) {
    // gets the distance from the ultrasonic sensor
    //set the trigger pin on one for 10 usec and then process to get the data
    Timer t;
    float distance;
    float duration; 
    
    //sends a trigger of 10 us 
    trigPin = 0; 
    wait_us(2);
    trigPin = 1;
    wait_us(10);
    trigPin = 0; 

    //measure the time taken from the echopin to stay high
    while (!echopin);         // read the duration time
    t.start();
    while (echopin){if(t.read() >= 10000){break;}}
    t.stop();
    
    duration = t.read_us(); 
    //pc.printf("1 : %f ", duration);
    distance = (duration/2) / 29.1;  
    t.reset();
    
    //printf("distance: %d\n",distance); //read the distance
    return distance;
}


void US_inter(){
    //trigger a flag to start the US routine
    US_flag=1;
    }
    
void ColourCheck(){
    
    ColourSensor.sensorOneCheck = ColourSensor.inOne.read();
    ColourSensor.sensorTwoCheck = ColourSensor.inTwo.read();
    
    bool sensorColourCheck = ColourSensor.sensorOneCheck & ColourSensor.sensorTwoCheck;
    
   if( ColourSensor.sensorColour == sensorColourCheck)ColourSensor.toggleC = 1;
   else ColourSensor.toggleB = 0;
   detectColour.attach(callback(&ColourSensor, &ColourSensor::readSensor), 0.05);   // Set call rate of colour detection function
   
   }



