#ifndef __mbed_irc_h__
#define __mbed_irc_h__

#include "mbed.h"
#include "EthernetInterface.h"
#include <vector>

/** 
 * IrcBot responds to commands on IRC using message users' handlers
 */ 
class IrcMessage {
    public:
        /** Create empty IrcMessage */
        IrcMessage();
        
        /** Create an IrcMessage
         * @param from The user sending the message.
         * @param to The user/channel receiving the message.
         * @param message The message.
         */ 
        IrcMessage(char * from, char * to, char * message);    
        char from[32], to[32], msg[256];
};

/**
 * Base MessageHandler class.
 * Users should write classes inheriting from MessageHandler to parse and 
 * respond to incoming IRC messages.
 */
class MessageHandler {
    public:
        MessageHandler() {};
        virtual IrcMessage handle(IrcMessage msg) {return IrcMessage();}
};

/**
 * IrcBot connects to an IRC network and joins a channel. Users can add message
 * handlers which parse incoming private messages and respond to them.
 */
class IrcBot {
    public:
        /** Create an IrcBot
         *
         * @param nickname Bot's nickname
         * @param network IRC network to join
         * @param port Port to connect to network on
         * @param channel Channel to connect to
         */
        IrcBot(char * nickname, char * network, int port, char * channel);
        /** Connect to the network. 
         *
         * Users should have already created a network interface 
         * (Ethernet/Wifi/3G/whatever) to carry the connection.
         */ 
        void connect();
        /// Disconnect from the network.
        void disconnect();
        /// Add a handler for incoming messages.
        void add(MessageHandler *);
        
        /** Read data from internet connection, parse input and handle any
         * incoming private messages
         */
        bool read();
    private:
        void handle(IrcMessage);
        void parse();
        void send(char *);
        Serial pc;
        TCPSocketConnection sock;
        char network[64];
        char channel[64];
        int port;
        char nickname[64];
        char password[64];
        bool ident, connected, setup, joined;
        char readbuffer[512];
        char parsebuffer[512];
        int parseindex;
        vector<MessageHandler *> handlers;
};

#endif