/*
* -------------------------------
* ConnectorBoxFW 2.0 (2016-09-26)
* -------------------------------
* Firmware for ConnectorBox TPH 1.0 (NXP LPC1768).
*
* Copyright (c) 2016, Martin Wolker (neolker@gmail.com)
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are met:
*     - Redistributions of source code must retain the above copyright
*       notice, this list of conditions and the following disclaimer.
*     - Redistributions in binary form must reproduce the above copyright
*       notice, this list of conditions and the following disclaimer in the
*       documentation and/or other materials provided with the distribution.
*     - Neither the name of Martin Wolker nor the
*       names of its contributors may be used to endorse or promote products
*       derived from this software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL MARTIN WOLKER BE LIABLE FOR ANY
* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
* ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "mbed.h"
#include "USBSerial.h"

#define ROT_SPEED           19200
#define ROT_TIMEOUT         250
#define ROT_INTERVAL        350
#define TEMPERATURE_START   29
#define TEMPERATURE_END     35
#define HUMIDITY_START      12
#define HUMIDITY_END        18
#define SERIAL_START        77
#define SERIAL_END          85
#define LED_DELAY           100

USBSerial       PC;
Serial          ROT_01(p28, p27);
Serial          ROT_02(p9, p10);
DigitalInOut    ID_01(p11);
DigitalInOut    ID_02(p12);
DigitalInOut    ID_03(p13);
DigitalInOut    ID_04(p14);
DigitalInOut    ID_05(p21);
DigitalInOut    ID_06(p22);
DigitalInOut    ID_07(p25);
DigitalInOut    ID_08(p26);
DigitalInOut    ID_09(p5);
DigitalInOut    ID_10(p6);
DigitalInOut    ID_11(p7);
DigitalInOut    ID_12(p8);
DigitalInOut    ID_13(p29);
DigitalInOut    ID_14(p30);
DigitalInOut    ID_15(p18);
DigitalInOut    ID_16(p17);
DigitalInOut    ID_17(p16);
DigitalInOut    ID_18(p15);
DigitalInOut    ID_19(p20);
DigitalInOut    ID_20(p19);
DigitalOut      LED(LED1);

enum DELAY {A = 6, B = 64, C = 60, D = 10, E = 9, F = 55, G = 0, H = 480, I = 70, J = 410};

void GetHelp()
{
    PC.printf("+-----------------------------------+\r\n");
    PC.printf("|      HELP - List of commands      |\r\n");
    PC.printf("+-----------------------------------+\r\n");
    PC.printf("| a -> Read all IDs and all ROTs    |\r\n");
    PC.printf("| d -> Debug of ROT_01 (RAW DATA)   |\r\n");
    PC.printf("| h -> Get this HELP                |\r\n");
    PC.printf("| i -> Read all IDs                 |\r\n");
    PC.printf("| p -> Periodic reading of all ROTs |\r\n");
    PC.printf("| r -> Read all ROTs                |\r\n");
    PC.printf("| s -> Signalization by USB_LED     |\r\n");
    PC.printf("| t -> Test of ID_01 and ROT_01     |\r\n");
    PC.printf("| v -> Get version of firmware      |\r\n");
    PC.printf("+-----------------------------------+\r\n");
}

void Blink(int info)
{
    if (info) PC.printf("LED blinking\r\n");
    for (int i = 0; i < 2; i++) {
        LED = 1;
        wait_ms(LED_DELAY);
        LED = 0;
        wait_ms(LED_DELAY);
    }
}

int Reset(DigitalInOut& pin)
{
    pin.output();
    pin = 0;
    wait_us(H);
    pin.input();
    wait_us(I);
    uint32_t result = pin;
    wait_us(J);
    return result;
}

void WriteBit(DigitalInOut& pin, uint32_t bit)
{
    pin.output();
    if (bit) {
        pin = 0;
        wait_us(A);
        pin.input();
        wait_us(B);
    } else {
        pin = 0;
        wait_us(C);
        pin.input();
        wait_us(D);
    }
}

uint32_t ReadBit(DigitalInOut& pin)
{
    uint32_t bit_value;
    pin.output();
    pin = 0;
    wait_us(A);
    pin.input();
    wait_us(E);
    bit_value  = pin;
    wait_us(F);
    return bit_value;
}

void WriteByte(DigitalInOut& pin, uint32_t byte)
{
    for (uint32_t bit = 0; bit < 8; ++bit) {
        WriteBit(pin, byte & 0x01);
        byte >>= 1;
    }
}

uint32_t ReadByte(DigitalInOut& pin)
{
    uint32_t byte = 0;
    for (uint32_t bit = 0; bit < 8; ++bit) {
        byte |= (ReadBit(pin) << bit);
    }
    return byte;
}

void GetID(DigitalInOut& pin, int number)
{
    Reset(pin);
    WriteByte(pin, 0x33);
    int B8 = ReadByte(pin);
    int B7 = ReadByte(pin);
    int B6 = ReadByte(pin);
    int B5 = ReadByte(pin);
    int B4 = ReadByte(pin);
    int B3 = ReadByte(pin);
    int B2 = ReadByte(pin);
    int B1 = ReadByte(pin);
    PC.printf("ID_%02d: %02X%02X%02X%02X%02X%02X%02X%02X\r\n", number, B1, B2, B3, B4, B5, B6, B7, B8);
}

void GetROT(Serial& port, int number, int debug)
{
    Timer t;
    int position = 0;
    char symbol = '0';
    char buffer[255] = "";
    t.start();
    port.printf("{F99RDD-\r");
    while((symbol != '\r') && (t.read_ms() < ROT_TIMEOUT)) {
        if(port.readable()) {
            symbol = port.getc();
            buffer[position] = symbol;
            position++;
        }
    }
    t.stop();
    if (debug) {
        PC.printf("------DEBUG_START-------\r\n");
        PC.printf("Reading time is: %d ms\r\n", t.read_ms());
        PC.printf("Count of characters: %d\r\n", position);
        PC.printf("ROT_%02d (RAW DATA):\r\n", number);
        if (position == 0) {
            PC.printf("N/A\r\n");
        } else {
            for (int d = 0; d < 105; d++) {
                PC.printf("%c", buffer[d]);
            }
            PC.printf("\n");
        }
        PC.printf("ROT_%02d (PARSED DATA):\r\n", number);
    }
    t.reset();
    if ((position == 105) && (buffer[16] != '-')) {
        PC.printf("ROT_%02d: ", number);
        for (int i = TEMPERATURE_START; i < TEMPERATURE_END; i++) {
            if (buffer[i] == '.') {
                PC.printf(",");
            } else {
                PC.printf("%c", buffer[i]);
            }
        }
        PC.printf("; ");
        for (int j = HUMIDITY_START; j < HUMIDITY_END; j++) {
            if (buffer[j] == '.') {
                PC.printf(",");
            } else {
                PC.printf("%c", buffer[j]);
            }
        }
        PC.printf("; ");
        for (int k = SERIAL_START; k < SERIAL_END; k++) {
            PC.printf("%c", buffer[k]);
        }
        PC.printf("\r\n");
    } else {
        PC.printf("ROT_%02d:   0,00;   0,00; 00000000\r\n", number);
    }
    if (debug) PC.printf("-------DEBUG_END--------\r\n");
}

void GetVersion()
{
    PC.printf("+--------------------------------+\r\n");
    PC.printf("|      ConnectorBox TPH 1.0      |\r\n");
    PC.printf("+--------------------------------+\r\n");
    PC.printf("| FW version: 2.0 (2016-09-26)   |\r\n");
    PC.printf("| Author:     Martin Wolker      |\r\n");
    PC.printf("| Contact:    neolker@gmail.com  |\r\n");
    PC.printf("+--------------------------------+\r\n");
}

int main(void)
{
    LED = 0;
    int flag = 1;
    ROT_01.baud(ROT_SPEED);
    ROT_02.baud(ROT_SPEED);
    Blink(0);
    wait_ms(100);
    while(1) {
        if (PC.readable()) {
            switch(PC.getc()) {
                case 'A':
                case 'a':
                    GetID(ID_01, 1);
                    GetID(ID_02, 2);
                    GetID(ID_03, 3);
                    GetID(ID_04, 4);
                    GetID(ID_05, 5);
                    GetID(ID_06, 6);
                    GetID(ID_07, 7);
                    GetID(ID_08, 8);
                    GetID(ID_09, 9);
                    GetID(ID_10, 10);
                    GetID(ID_11, 11);
                    GetID(ID_12, 12);
                    GetID(ID_13, 13);
                    GetID(ID_14, 14);
                    GetID(ID_15, 15);
                    GetID(ID_16, 16);
                    GetID(ID_17, 17);
                    GetID(ID_18, 18);
                    GetID(ID_19, 19);
                    GetID(ID_20, 20);
                    GetROT(ROT_01, 1, 0);
                    GetROT(ROT_02, 2, 0);
                    Blink(0);
                    break;
                case 'D':
                case 'd':
                    GetROT(ROT_01, 1, 1);
                    Blink(0);
                    break;
                case 'H':
                case 'h':
                    GetHelp();
                    Blink(0);
                    break;
                case 'I':
                case 'i':
                    GetID(ID_01, 1);
                    GetID(ID_02, 2);
                    GetID(ID_03, 3);
                    GetID(ID_04, 4);
                    GetID(ID_05, 5);
                    GetID(ID_06, 6);
                    GetID(ID_07, 7);
                    GetID(ID_08, 8);
                    GetID(ID_09, 9);
                    GetID(ID_10, 10);
                    GetID(ID_11, 11);
                    GetID(ID_12, 12);
                    GetID(ID_13, 13);
                    GetID(ID_14, 14);
                    GetID(ID_15, 15);
                    GetID(ID_16, 16);
                    GetID(ID_17, 17);
                    GetID(ID_18, 18);
                    GetID(ID_19, 19);
                    GetID(ID_20, 20);
                    Blink(0);
                    break;
                case 'P':
                case 'p':
                    PC.printf("-------------START--------------\r\n");
                    while(flag) {
                        GetROT(ROT_01, 1, 0);
                        GetROT(ROT_02, 2, 0);
                        Blink(0);
                        wait_ms(ROT_INTERVAL);
                        if (PC.readable()) {
                            PC.getc();
                            PC.printf("--------------END---------------\r\n");
                            flag = 0;
                        }
                    }
                    flag = 1;
                    break;
                case 'R':
                case 'r':
                    GetROT(ROT_01, 1, 0);
                    GetROT(ROT_02, 2, 0);
                    Blink(0);
                    break;
                case 'S':
                case 's':
                    Blink(1);
                    break;
                case 'T':
                case 't':
                    GetID(ID_01, 1);
                    GetROT(ROT_01, 1, 0);
                    Blink(0);
                    break;
                case 'V':
                case 'v':
                    GetVersion();
                    Blink(0);
                    break;
                default:
                    PC.printf("Unknown command, try \"h\" for HELP...\r\n");
                    Blink(0);
                    break;
            }
        }
    }
}
