#ifndef CATCH_MODEL_H
#define CATCH_MODEL_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Basket.h"
#include "Objects.h"

/** Catch_Model Class
@brief Class brings together Basket, Objects and Fruit to create 'Fruit Basket'
@brief game. This class also uses variables from the other classes to display
@brief information at the top of the screen i.e. score, remaining lives and an
@brief indicator showing if the player can use one of the available 'power-ups'.

@author Nathan Johnston
@date 14th March 2017
*/

class Catch_Model
{
public:

    Catch_Model();
    ~Catch_Model();

    //INITILISATION FUNCTIONS//

    /** Initialise Game
     *
     *   Main initialisation function. The parameters are used to call
     *   Basket::init(int y, int width) and Objects::init(int speed), as well
     *   as set the number of lives/misses the player will be allowed and set
     *   the default value of a variable that restricts how often the player
     *   can use the A and B buttons.
     *   @param basket_y - y co-ordinate of the basket (0 to 47)
     *   @param basket_width - width of the basket (0 to 83)
     *   @param objects_speed - "fall" speed of the object (2,3,4,5)
     *   @param lives - number of lives/misses the player has
     */
    void init(int basket_y, int basket_width, int objects_speed, int lives);


    //UPDATE FUNCTIONS//

    /** Read Joystick Input
     *
     *   Read the direction and magnitude of the joystick and assign
     *   the values to variables by calling functions from the Gamepad
     *   library.
     *   @param pad - Gamepad custom library
     */
    void input(Gamepad &pad);

    /** Update Game
     *
     *   Checks if an object is caught by the basket or not then increases
     *   score or reduces lives appropriately. Moves the basket with either
     *   joystick or buttons and moves the objects too.
     *   @param lcd - N5110 custom library
     *   @param pad - Gamepad custom library
     */
    void update(N5110 &lcd, Gamepad &pad);


    //GAME RULES FUNCTIONS//

    /** Check Object has been Caught
     *
     *   Gets the x and y reference co-ordinates of the basket and the
     *   falling object from their respective libraries; if the x co-ordinate
     *   of the object is within the width of the basket and the y co-ordinate
     *   of the object is greater than that of the basket the object will be
     *   undrawn, the appropriate score will be added and the object will be
     *   re-initialised.
     *   @param lcd - N5110 custom library
     *   @param pad - Gamepad custom library
     */
    void check_basket_catch(N5110 &lcd, Gamepad &pad);

    /** Check Object has Missed
     *
     *   Gets the x and y reference co-ordinates of the basket and the
     *   falling object from their respective libraries; if the x co-ordinate
     *   of the object is outside the width of the basket and the y co-ordinate
     *   of the object is greater than that of the basket the object will be
     *   undrawn, the number of lives will be reduced and the object will be
     *   re-initialised.
     *   @param lcd - N5110 custom library
     *   @param pad - Gamepad custom library
     */
    void check_basket_miss(N5110 &lcd, Gamepad &pad);

    /** Add Correct Score
     *
     *   Gets the value of a variable from Objects that is specific to each
     *   object (type of fruit) and uses it to call the appropriate add_score
     *   function from Basket; increasing the score by 1, 2, 5 or 10.
     */
    void add_score();

    /** Get Lives
     *
     *   Return the remaining number of lives available to the player.
     */
    int get_lives();


    //BUTTON FUNCTIONS//

    /** Check A Button
     *
     *   If the A button is pressed, call Objects::undraw(N5110 &lcd) wherever
     *   it is on the display and call Objects::init(int speed) to give
     *   the player a second chance. Also set the value of a variable to zero
     *   so that the function cannot be recalled straight away.
     *   @param lcd - N5110 custom library
     *   @param pad - Gamepad custom library
     */
    void check_a(N5110 &lcd, Gamepad &pad);
    
    /** Check B Button
     *
     *   If the B button is pressed, increase number of lives by 1. Also set 
     *   the value of a variable to zero so that the function cannot be 
     *   recalled straight away.
     *   @param lcd - N5110 custom library
     *   @param pad - Gamepad custom library
     */
    void check_b(N5110 &lcd, Gamepad &pad);
    
    /** Set Delay to ON
     *
     *   10 seconds after the buttons A or B are pressed this function is called
     *   resetting the value of a variable to 1 so the buttons can be used again.
     */
    void set_delay();


    //DISPLAY FUNCTIONS//
    
    /** Draw All Features
     *
     *   Calls draw functions from the Basket and Objects libraries as well
     *   as functions to display the score, remaining lives and an indicator
     *   that tells the player if the A and B buttons are ready to be used.
     *   Once the game is finished, a 'Game Over' screen is displayed showing
     *   the final score.
     *   @param lcd - N5110 custom library
     *   @param pad - Gamepad custom library
     */
    void draw(N5110 &lcd, Gamepad &pad);
    
    /** Display the Number of Lives
     *
     *   Print a string of characters to the buffer, indicating how many lives
     *   remain.
     *   @param lcd - N5110 custom library
     */
    void print_lives(N5110 &lcd);
    
    /** Display the Score
     *
     *   Print a string of characters to the buffer, indicating the score.
     *   @param lcd - N5110 custom library
     */
    void print_score(N5110 &lcd);
    
    /** Display the Final Score
     *
     *   Print two strings of characters to the buffer, one indicating the 
     *   score and another saying 'Game Over'.
     *   @param lcd - N5110 custom library
     */
    void final_score(N5110 &lcd);
    
    /** Display the Powerup Indicator
     *
     *   Print a tick or cross to the buffer, indicating whether or not 
     *   the A and B buttons are ready to be used.
     *   @param lcd - N5110 custom library
     */
    void print_delay(N5110 &lcd);

private:

    //OBJECTS//
    Basket basket;
    Objects objects;
    Timeout timeout;

    //VARIABLES//
    int _basket_y;
    int _basket_width;
    int _objects_speed;
    int _lives;
    int _delay;

    //JOYSTICK PARAMETERS//
    Direction _d;
    float _mag;

};
#endif