#ifndef BASKET_H
#define BASKET_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

/** Basket Class
@brief Class that draws a basket on the display and allows it to be moved left
@brief or right along the bottom of the screen. If an object comes into contact 
@brief with the basket, points are scored.

@author Nathan Johnston
@date 14th March 2017
*/
class Basket
{
public:

    Basket();
    ~Basket();

    //INITILISATION FUNCTION//

    /** Initialise Basket
     *
     *   Sets reference points by which basket can be drawn on screen.
     *   Sets score equal to zero.
     *   @param y - y co-ordinate of the basket (0 to 47)(41 by default)
     *   @param width - width of the basket (0 to 83)(12 by default)
     */
    void init(int y, int width);


    //UPDATE FUNCTIONS//

    /** Move Basket w/ Joystick
     *
     *   Move the basket left/right using the gamepad's joystick.
     *   @param d - direction the joystick is pushed towards (N,NE,E,SE,S,SW,W,NW)
     *   @param mag - float in range 0.0 to 1.0
     */
    void move_stick(Direction d, float mag);

    /** Move the basket w/ L and R
     *
     *   Move the basket left/right using the gamepad's L and R buttons.
     *   @param pad - Gamepad custom library
     */
    void move_LR(Gamepad &pad);


    //SCORE FUNCTIONS//

    /** Increase Score (1)
     *
     *   Adds 1 to overall score.
     */
    void add_score_1();

    /** Increase Score (2)
     *
     *   Adds 2 to overall score.
     */
    void add_score_2();

    /** Increase Score (3)
     *
     *   Adds 5 to overall score.
     */
    void add_score_5();

    /** Increase Score (4)
     * 
     *   Adds 10 to overall score.
     */
    void add_score_10();

    /** Get Score
     *
     *   Returns value of score.
     */
    int get_score();


    //DISPLAY FUNCTIONS//

    /** Draw Basket
    *
    *   Draw the basket in the screen buffer using
    *   the reference points set in Basket::init().
    *   @param lcd - N5110 custom library
    */
    void draw(N5110 &lcd);

    /** Get x
    *
    *   Return value of the x co-ordinate (0 to 71).
    */
    int get_x();

    /** Get y
    *
    *   Return value of the y co-ordinate (0 to 47).
    */
    int get_y();

private:

    //VARIABLES//
    int y_ref;
    int x_ref;
    int basket_width;
    int stick_speed;
    int score;

};
#endif
