#ifndef _PLAYERENTITY_
#define _PLAYERENTITY_

#include "mbed.h"
#include "life_entity.h"

#define DEBUG_PLAYER 0x01


/** Player class derived from life_entity.
 * 
 * This class is derived from life_entity.
 *
 * Example:
 * @code
 *
 * // Create a new player object.
 * player *user = new player();
 *
 * int main()
 * {    
 *     // Get the user's level (for example)
 *     char playerLevel = user->getLevel();
 * }
 * @endcode
 */
class player : public life_entity
{

public:

    /** Player class constructor.
     * 
     * This class is derived from life_entity.
     *
     * Example:
     * @code
     *
     * // Create a new player object.
     * player *user = new player();
     *
     * int main()
     * {    
     *     // Get the user's level (for example)
     *     char playerLevel = user->getLevel();
     * }
     * @endcode
     */
    player();
    
    /** Get the number of experience points the player currently has.
    *
    * Example:
    * @code
    * // Create a new player object.
    * player *user = new player();
    *
    * int main()
    * {    
    *     // Get the user's xp
    *     unsigned long int xp = user->getExperience();
    * }
    * @endcode
    */
    unsigned long int getExperience();
    
    /** Get the number player's current level.
    *
    * Example:
    * @code
    * // Create a new player object.
    * player *user = new player();
    *
    * int main()
    * {    
    *     // Get the user's level
    *     char playerLevel = user->getLevel();
    * }
    * @endcode
    */
    char getLevel();
    
    /** Check to see if the player is ready to level up.
    *
    * Compare the player's current experience with the amount
    * of experience required to reach the next level.
    * If enough experience is present, level up the player.
    *
    * Example:
    * @code
    * // Create a new player object.
    * player *user = new player();
    *
    * int main()
    * {    
    *     // Check for level up
    *     user->isLevelUp();
    * }
    * @endcode
    */
    void isLevelUp();
    
    /** Add experience points (and apply rest health).
    *
    * Add experience points for the win and calculate
    * bonus health from 'rest'.
    *
    * Example:
    * @code
    * // Create a new player object.
    * player *user = new player();
    *
    * int main()
    * {    
    *     // Add experience points to player
    *     user->addExperience();
    * }
    * @endcode
    */
    void addExperience();
    
    /** Display player experience ramp.
    *
    * Display the amount of experience required
    * for each level.
    *
    * Example:
    * @code
    * // Create a new player object.
    * player *user = new player();
    *
    * int main()
    * {    
    *     // Print experience ramp
    *     user->displayExpRamp();
    * }
    * @endcode
    */
    void displayExpRamp();
    
    /** Take damage from apponent.
    *
    * This applies the enemy's damage roll to the player's health after a dodge roll
    * is performed.
    * 
    * @param roll The damage roll to be passed in (most likely from rollDamage()).
    *
    * Example:
    * @code
    * player *user = new player();
    * life_entity *currentEnemy; 
    *
    * int main()
    * {    
    *     // Use the pointer to create a new armoured_vehicle (derived class)
    *     currentEnemy = new enemy(user);
    *     
    *     // Enemy rolls for damage
    *     int roll = currentEnemy->rollDamage();
    *     // Player takes the hit
    *     user->takeDamage(roll);
    * }
    * @endcode
    */
    virtual void takeDamage(int roll);
    
    /** Send information about the current enemy.
    *
    * Pass amount of health and (optionally) level of current enemy.
    *
    * @param health Amount of enemy health left.
    * @param level Enemy level (optional).  Defaults to 0x00.
    *
    * Example:
    * @code
    * // Create a new player object.
    * player *user = new player();
    *
    * int main()
    * {    
    *       // Get current enemy health
    *       int enemyHealth = currentEnemy->getHealth();
    *       // Send update on enemy health status to user
    *       user->setCurrentEnemy(enemyHealth);
    * }
    * @endcode
    */
    void setCurrentEnemy(int health, char level = 0x00);
    
    /** Check to see if the player is dead.
    *
    * If player is dead it makes fun of player and pretends to restart everything.
    * Returns true if player is dead.
    *
    * Example:
    * @code
    * // Create a new player object.
    * player *user = new player();
    *
    * int main()
    * {    
    *       char thereIsCake = user->isDead();
    *       if(thereIsCake)
    *       {
    *           // Do your delete and new code here
    *       }
    * }
    * @endcode
    */
    char isDead();
        
protected:
    /**
    * Variable to hold amount of player's experience points.
    */
    unsigned long int _experience; 
    
    /**
    * Variable to hold the current enemy's level.
    */
    char _enemyLevel;
    /**
    * Variable to hold the current enemy's health.
    */
    int _enemyHealth;
    
    /**
    * Base health of player (that is then multiplied by level).
    */
    int _baseHealth;
};

#endif