//
// Test program for the CC1200 morse code example
//

#include <mbed.h>
#include <SerialStream.h>

#include <CC1200.h>
#include <cinttypes>

#include "CC1200Morse.h"

#define PIN_SPI_MOSI P0_9
#define PIN_SPI_MISO P0_8
#define PIN_SPI_SCLK P0_7

#define PIN_CC1200_CS P0_6
#define PIN_CC1200_RST P0_29

BufferedSerial serial(USBTX, USBRX, 115200);
SerialStream<BufferedSerial> pc(serial);

CC1200 radio(PIN_SPI_MOSI, PIN_SPI_MISO, PIN_SPI_SCLK, PIN_CC1200_CS, PIN_CC1200_RST, &pc);

void testMorseCodeConversion()
{
	radio.begin();
	CC1200Morse morseConverter(radio);

	const size_t bufferLen = 64;
	uint8_t outputBuffer[bufferLen];

	char const * testString = "eternal silence@54!";

	CC1200Morse::EncodedMorse morse = morseConverter.convertToMorse(testString, outputBuffer, bufferLen);

	if(!morse.valid)
	{
		pc.printf("Morse is invalid\n");
	}
	else
	{
		pc.printf("Output %zu bytes:", morse.totalLength);
		for(size_t index = 0; index < morse.totalLength; ++index)
		{
			pc.printf(" %" PRIx8, outputBuffer[index]);
		}
		pc.printf("\n");
	}
}

/**
 * Test sending some arbitrary bytes as OOK modulated data.
 */
void testMorseByteTransmission()
{
	const float timeUnit = 0.1f;

	radio.begin();
	CC1200Morse morseConverter(radio);
	morseConverter.configure(CC1200::Band::BAND_820_960MHz, 915e6f, timeUnit, 14.5);

	const char testString[] = "\xFF\x0E\xFF";
	const size_t testStringLength = 3;

	// manually create morse code data
	CC1200Morse::EncodedMorse morse;
	morse.valid = true;
	morse.buffer = reinterpret_cast<uint8_t const *>(testString);
	morse.byteLen = testStringLength;
	morse.bitLen = 0;
	morse.totalLength = testStringLength;

	morseConverter.transmit(morse);

	Timer messageTimer;
	messageTimer.start();

	radio.setOnTransmitState(CC1200::State::IDLE);
	radio.startTX();

	// wait until all bytes have been transmitted.
	// Note: the FIFO length is 0 when the last byte is being sent, so we can't just check the FIFO length
	while(radio.getTXFIFOLen() > 0 || radio.getState() != CC1200::State::IDLE)
	{
		pc.printf("TX FIFO size: %zu\n", radio.getTXFIFOLen());
		ThisThread::sleep_for(std::chrono::milliseconds(static_cast<uint32_t>(timeUnit * 1000)));
		//ThisThread::sleep_for(1ms);
	}

	float timeSeconds = std::chrono::duration_cast<std::chrono::duration<float>>(messageTimer.elapsed_time()).count();
	size_t numBits = ((morse.byteLen * 8) + morse.bitLen);
	float effectiveBitrate = numBits / timeSeconds;

	pc.printf("Sent %zu bits in %.03f s, effective bitrate = %.03f sps\n", numBits, timeSeconds, effectiveBitrate);

}

void testMorseCodeTransmission()
{
	const float timeUnit = 0.1f;

	radio.begin();
	CC1200Morse morseConverter(radio);
	morseConverter.configure(CC1200::Band::BAND_820_960MHz, 915e6f, timeUnit, 0);

	const size_t bufferLen = 64;
	uint8_t outputBuffer[bufferLen];

	char const * testString = "eternal silence@54!";

	CC1200Morse::EncodedMorse morse = morseConverter.convertToMorse(testString, outputBuffer, bufferLen);

	if(!morse.valid)
	{
		pc.printf("Morse is invalid\n");
	}

	morseConverter.transmit(morse);

	radio.setOnTransmitState(CC1200::State::IDLE);
	radio.startTX();

	// wait until all bytes have been transmitted.
	// Note: the FIFO length is 0 when the last byte is being sent, so we can't just check the FIFO length
	while(radio.getTXFIFOLen() > 0 || radio.getState() != CC1200::State::IDLE)
	{
		pc.printf("TX FIFO size: %zu\n", radio.getTXFIFOLen());
		ThisThread::sleep_for(std::chrono::milliseconds(static_cast<uint32_t>(timeUnit * 1000)));
		//ThisThread::sleep_for(1ms);
	}
}

int main()
{
	pc.printf("\nCC1200 Morse Test Suite:\n");

	while(1){
		int test=-1;
		//MENU. ADD AN OPTION FOR EACH TEST.
		pc.printf("Select a test: \n");
		pc.printf("1.  Test converting to Morse code\n");
		pc.printf("2.  Test transmitting bytes as Morse\n");
		pc.printf("3.  Test transmitting string of Morse\n");

		pc.scanf("%d", &test);
		printf("Running test %d:\n\n", test);
		//SWITCH. ADD A CASE FOR EACH TEST.
		switch(test) {
			case 1:			testMorseCodeConversion();			break;
			case 2:			testMorseByteTransmission();		break;
			case 3:			testMorseCodeTransmission();		break;
			default:        pc.printf("Invalid test number. Please run again.\n"); continue;
		}
		pc.printf("done.\r\n");
	}

	return 0;

}
