#include "ChemIDMeasurer.h"
#include <cinttypes>

namespace mbed
{
    FileHandle *mbed_override_console(int)
    {
        static BufferedSerial serial(USBTX, USBRX, 115200);
        return &serial;
    }
}

ChemIDMeasurer::ChemIDMeasurer():
soc(I2C_SDA, I2C_SCL, 100000)
{
}

void ChemIDMeasurer::activateCharger()
{
	// If you have code to activate the charger, put it here
}

void ChemIDMeasurer::deactivateCharger()
{
	// If you have code to deactivate the charger, put it here
}

void ChemIDMeasurer::activateLoad()
{
	// If you have code to activate the dummy load, put it here
}

void ChemIDMeasurer::deactivateLoad()
{
	// If you have code to deactivate the dummy load, put it here
}

void ChemIDMeasurer::setState(ChemIDMeasurer::State newState)
{
	state = newState;
	stateTimer.reset();
}

void ChemIDMeasurer::runMeasurement()
{
	totalTimer.start();
	stateTimer.start();

	while(state != State::DONE)
	{
		// read data
		uint16_t voltage_mV = soc.getVoltage();
		int32_t current_mA = soc.getCurrent();
		double temperatureC = soc.getTemperature();
		char const * comment = "";

		// update based on state
		switch (state)
		{
			case State::INIT:
				// Print header
				printf("Elapsed Time (s), Voltage (mV), Current (mA), Temperature (deg C), SoC (%%), Comments\n");
				setState(State::CHARGE);
				activateCharger();
				comment = "Activating charger and entering CHARGE";
				break;

			case State::CHARGE:
				// threshold current = C/10
				if(current_mA < DESIGNCAP/10)
				{
					deactivateCharger();
					setState(State::RELAX_CHARGED);
					comment = "Deactivating charger and entering RELAX_CHARGED";
				}
				break;

			case State::RELAX_CHARGED:
				if(stateTimer.elapsed_time() > 2h)
				{
                    activateLoad();
					setState(State::DISCHARGE);
					comment = "Done relaxing and entering discharge -- please disconnect charger and connect C/10 load now.";
				}
				break;

			case State::DISCHARGE:
				// Change states once we hit the termination voltage
				if(voltage_mV < ZEROCHARGEVOLT * CELLCOUNT)
				{
                    deactivateLoad();
					setState(State::RELAX_DISCHARGED);
					comment = "Done discharging -- please remove C/10 load now.";
				}
				// BQ34Z100 reports positive current always, but TI's tool expects discharging to
				// be negative current.
				current_mA *= -1;
				break;

			case State::RELAX_DISCHARGED:
				if(stateTimer.elapsed_time() > 5h)
				{
					setState(State::DONE);
					comment = "Done!";
				}
				break;

			default:
				// will never be hit but here to silence warning
				break;
		}


		// print data column
		printf("%" PRIi64 ", %" PRIi16 ", %" PRIi32 ", %f, %" PRIu8 ", %s\n",
			std::chrono::duration_cast<std::chrono::seconds>(totalTimer.elapsed_time()).count(),
			voltage_mV, current_mA, temperatureC, soc.getSOC(), comment);

		// wait, update freq is every 5 seconds
		ThisThread::sleep_for(5s);
	}
}

ChemIDMeasurer measurer;

int main()
{
	measurer.runMeasurement();
	return 0;
}