#ifndef BNO080_BNO080ASYNC_H
#define BNO080_BNO080ASYNC_H

#if MBED_CONF_RTOS_PRESENT

#include <BNO080.h>
#include <Mutex.h>

/**
 * Asynchronous version of the SPI BNO080 driver.
 * Since the timing requirements of this driver are so different,
 * wouldn't it be great to have a dedicated thread to handle them?
 * This class provides exactly that, using an internal thread triggered
 * by interrupts to connect to the BNO.
 *
 * Note: the internal thread is started the first time begin() is called,
 * and is shut down when the class is destroyed.
 */
class BNO080Async : public BNO080SPI
{
public:
	// Mutex protecting all sensor data in the driver.
	// You must lock this while reading data and unlock it when done.
	// While this is locked, the background thread is prevented from running.
	Mutex bnoDataMutex;

private:
	// thread in charge of communicating with the BNO
	Thread commThread;

	// EventFlags to allow signalling the thread to wake up
	EventFlags wakeupFlags;

	// main function for the thread.
	// Handles starting the comm loop.
	void threadMain();

	// Code loop to communicate with the BNO.
	// Runs forever in normal operation.
	// Returns true to request a restart.
	// Returns false to request the thread to shutdown.
	bool commLoop();

	// flag to indicate that we have valid data to send queued in the TX buffer.
	// Protected by bnoDataMutex.
	bool dataToSend = false;

	// data for packet to send, if above flag is true
	uint8_t txPacketChannelNumber;
	uint8_t txPacketDataLength;

	// Condition variable signaled whenever a packet is sent.
	ConditionVariable dataSentCV;

	// flag used for updateData() return value.
	// True if any data packets have been received since the last update.
	bool dataReceived = false;

	// true iff the thread is currently in the middle of a TX/RX operation.
	// Causes _int interrupts (which get generated by the comms process) to be ignored.
	bool inTXRX = false;

	bool sendPacket(uint8_t channelNumber, uint8_t dataLength) override;

	void clearSendBuffer() override;

	// waiting for packet state info
	bool waitingForPacket = false;
	uint8_t waitingForChannel;
	uint8_t waitingForReportID;
	bool waitingPacketArrived = false;
	bool clearToRxNextPacket = false;

	ConditionVariable waitingPacketArrivedCV;
	ConditionVariable waitingPacketProcessedCV;

	bool waitForPacket(int channel, uint8_t reportID, std::chrono::milliseconds timeout = 125ms) override;

public:
	/**
	 * Construct a BNO080Async.
	 * This doesn't actually initialize the chip, you will need to call begin() for that.
	 *
	 * NOTE: while some schematics tell you to connect the BOOTN pin to the processor, this driver does not use or require it.
	 * Just tie it to VCC per the datasheet.
	 *
	 * @param debugPort Serial port to write output to.  Cannot be nullptr.
	 * @param rstPin Hardware reset pin, resets the IMU
	 * @param intPin Hardware interrupt pin, this is used for the IMU to signal the host that it has a message to send
	 * @param wakePin Hardware wake pin, this is used by the processor to signal the BNO to wake up and receive a message
	 * @param misoPin SPI MISO pin
	 * @param mosiPin SPI MOSI pin
	 * @param sclkPin SPI SCLK pin
	 * @param csPin SPI CS pin
	 * @param spiSpeed SPI frequency.  The BNO's max is 3MHz.
	 * @param threadPriority Priority to give the internal thread.  Defaults to AboveNormal so it will run whenever it can.
	 */
	BNO080Async(Stream *debugPort, PinName rstPin, PinName intPin, PinName wakePin, PinName misoPin, PinName mosiPin, PinName sclkPin, PinName csPin, int spiSpeed=3000000, osPriority threadPriority=osPriorityAboveNormal);

	/**
	 * Resets and connects to the IMU.  Verifies that it's connected, and reads out its version
	 * info into the class variables above.
	 *
	 * Async version also starts the internal communication thread, restarting it if it's
	 * already started.
	 *
	 * If this function is failing, it would be a good idea to turn on BNO_DEBUG in the cpp file to get detailed output.
	 *
	 * @return whether or not initialization was successful
	 */
	bool begin() override;

	/**
	 * In BNO080Async, there is no need to call updateData() in order to get new
	 * results, but this call is provided for compatibility.
	 *
	 * It maintains its behavior of returning true iff packets have been received since the last call.
	 * You must lock bnoDataMutex to call this.
	 * @return
	 */
	bool updateData() override;

	/**
	 * Locks the data mutex.
	 */
	void lockMutex() override
	{
		bnoDataMutex.lock();
	}

	/**
	 * Unlocks the data mutex.
	 */
	void unlockMutex() override
	{
		bnoDataMutex.unlock();
	}
};

#endif


#endif //MBED_CMAKE_TEST_PROJECT_BNO080ASYNC_H
