/*
 * Copyright (C)2010-2012 Henning Karlsen. All right reserved.
 * Copyright (C)2012 Todor Todorov.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to:
 *
 * Free Software Foundation, Inc.
 * 51 Franklin St, 5th Floor, Boston, MA 02110-1301, USA
 *
 *********************************************************************/
#include "st775s9bit.h"
#include "helpers.h"

HX8340S_LCD::HX8340S_LCD( PinName CS, PinName RESET, PinName SCL, PinName SDI, PinName BL, backlight_t blType, float defaultBackLightLevel )
    : LCD( 160, 128, CS, NC, RESET, BL, blType, defaultBackLightLevel ), _lcd_pin_scl( SCL ), _lcd_pin_sdi( SDI )
{
}

void HX8340S_LCD::Initialize( orientation_t orientation, colordepth_t colors )
{
    _orientation = orientation;
    _colorDepth = colors;

    wait_ms( 100 );
    _lcd_pin_reset = HIGH;
    wait_ms( 5 );
    _lcd_pin_reset = LOW;
    wait_ms( 15 );
    _lcd_pin_reset = HIGH;
    _lcd_pin_cs = HIGH;
    _lcd_pin_scl = HIGH;
    _lcd_pin_sdi = HIGH;
    if ( _lcd_pin_bl != 0 )
        *_lcd_pin_bl = HIGH;
    else if ( _bl_pwm != 0 )
        *_bl_pwm = _bl_pwm_default;
    wait_ms( 55 );

    Activate();

    WriteCmd( 0x11 ); //Sleep out
    wait_ms( 160 );  //Delay 120ms

    WriteCmd( 0x3A );  //05: 65K (16 bit) 06: 262K (8 bit)
    WriteByteData( 0x05 );
    WriteCmd( 0x36 );
    WriteByteData( 0x08 );

    //ST7735S Frame rate setting
    WriteCmd( 0xB1 ); //Frame Rate Control (In normal mode/ Full colors)
    WriteByteData( 0x05 );
    WriteByteData( 0x3A );
    WriteByteData( 0x3A );

    WriteCmd( 0xB2 ); //Frame Rate Control (In Idle mode/ 8-colors)
    WriteByteData( 0x05 );
    WriteByteData( 0x3A );
    WriteByteData( 0x3A );

    WriteCmd( 0xB3 ); //Frame Rate Control (In Partial mode/ full colors)
    WriteByteData( 0x05 );
    WriteByteData( 0x3A );
    WriteByteData( 0x3A );
    WriteByteData( 0x05 );
    WriteByteData( 0x3A );
    WriteByteData( 0x3A );

    WriteCmd( 0xB4 ); //Dot Inversion
    WriteByteData( 0x03 );

    //ST7735S Power setting

    WriteCmd( 0xC0 ); //Power Control 1
    WriteByteData( 0x62 );
    WriteByteData( 0x02 );
    WriteByteData( 0x04 );


    WriteCmd( 0xC1 );  //Power Control 2
    WriteByteData( 0xC0 );


    WriteCmd( 0xC2 ); //Power Control 3 (in Normal mode/ Full colors)
    WriteByteData( 0x0D );
    WriteByteData( 0x00 );


    WriteCmd( 0xC3 ); //Power Control 4 (in Idle mode/ 8-colors)
    WriteByteData( 0x8D );
    WriteByteData( 0x6A );


    WriteCmd( 0xC4 ); //Power Control 5 (in Partial mode/ full-colors)
    WriteByteData( 0x8D );
    WriteByteData( 0xEE );

    WriteCmd( 0xC5 ); //VCOM Control 1
    WriteByteData( 0x12 );

    //ST7735S Gamma Setting
    WriteCmd(0xE0); //Gamma (‘+’polarity) Correction Characteristics Setting
    WriteByteData(0x03);
    WriteByteData(0x1B);
    WriteByteData(0x12);
    WriteByteData(0x11);
    WriteByteData(0x3F);
    WriteByteData(0x3A);
    WriteByteData(0x32);
    WriteByteData(0x34);
    WriteByteData(0x2F);
    WriteByteData(0x2B);
    WriteByteData(0x30);
    WriteByteData(0x3A);
    WriteByteData(0x00);
    WriteByteData(0x01);
    WriteByteData(0x02);
    WriteByteData(0x05);

    WriteCmd(0xE1); //Gamma ‘-’polarity Correction Characteristics Setting
    WriteByteData(0x03);
    WriteByteData(0x1B);
    WriteByteData(0x12);
    WriteByteData(0x11);
    WriteByteData(0x32);
    WriteByteData(0x2F);
    WriteByteData(0x2A);
    WriteByteData(0x2F);
    WriteByteData(0x2E);
    WriteByteData(0x2C);
    WriteByteData(0x35);
    WriteByteData(0x3F);
    WriteByteData(0x00);
    WriteByteData(0x00);
    WriteByteData(0x01);
    WriteByteData(0x05);

    WriteCmd(0xFC); //Enable Gate power save mode
    WriteByteData(0x8C);
//--------------------------------------------------------------------//
    WriteCmd(0x2A); //Column Address Set
    WriteByteData(0x00);
    WriteByteData(0x00);//02 : old FPC | 00 : New FPC
    WriteByteData(0x00);//             |
    WriteByteData(0x7F);//129 : old FPC| 128 : New FPC

    WriteCmd(0x2B); //Row Address Set
    WriteByteData(0x00);
    WriteByteData(0x00);//01 : Old  | 00 : New FPC
    WriteByteData(0x00);//          |
    WriteByteData(0x9F);//160 : Old | 159 : New FPC
//--------------------------------------------------------------------//

    WriteCmd( 0x29 ); // DISPON
    WriteCmd(0x2C); //GRAM
    wait_ms( 10 );

    ClearXY();
    Deactivate();
}

void HX8340S_LCD::Sleep( void )
{
    Activate();
    WriteCmd( 0x28 );
    wait_ms( 10 );
    WriteCmd( 0x10 );
    wait_ms( 125 );
    LCD::Sleep();
    Deactivate();
}

void HX8340S_LCD::WakeUp( void )
{
    Activate();
    WriteCmd( 0x29 );
    wait_ms( 10 );
    WriteCmd( 0x11 );
    wait_ms( 125 );
    LCD::WakeUp();
    Deactivate();
}

void HX8340S_LCD::WriteCmd( unsigned short cmd )
{
    _lcd_pin_sdi = LOW;
    pulseLow( _lcd_pin_scl );
    serializeByte( cmd & 0xFF );
}

void HX8340S_LCD::WriteData( unsigned short data )
{
    _lcd_pin_sdi = HIGH;
    pulseLow( _lcd_pin_scl );
    serializeByte( ( data >> 8 ) & 0xFF );
    _lcd_pin_sdi = HIGH;
    pulseLow( _lcd_pin_scl );
    serializeByte( data & 0xFF );
}

void HX8340S_LCD::WriteByteData( unsigned char data )
{
    _lcd_pin_sdi = HIGH;
    pulseLow( _lcd_pin_scl );
    serializeByte( data );
}

void HX8340S_LCD::SetXY( unsigned short x1, unsigned short y1, unsigned short x2, unsigned short y2 )
{
    WriteCmdData( 0x2A, x1 );  // CASET
    WriteData( x2 );
    WriteCmdData( 0x2B, y1 );  // PASET
    WriteData( y2 );
    WriteCmd( 0x2C ); // RAMWR
}

void HX8340S_LCD::SetPixelColor( unsigned int color, colordepth_t mode )
{
    unsigned char r = 0, g = 0, b = 0;
    unsigned short clr;
    if ( _colorDepth == RGB16 ) {
        switch ( mode ) {
            case RGB16:
                WriteData( color & 0xFFFF );
                break;
            case RGB18:
                r = ( color >> 10 ) & 0xF8;
                g = ( color >> 4 ) & 0xFC;
                b = ( color >> 1 ) & 0x1F;
                clr = ( ( r | ( g >> 5 ) ) << 8 ) | ( ( g << 3 ) | b );
                WriteData( clr );
                break;
            case RGB24:
                r = ( color >> 16 ) & 0xF8;
                g = ( color >> 8 ) & 0xFC;
                b = color & 0xF8;
                clr = ( ( r | ( g >> 5 ) ) << 8 ) | ( ( g << 3 ) | ( b >> 3 ) );
                WriteData( clr );
                break;
        }
    } else if ( _colorDepth == RGB18 ) {
        switch ( mode ) {
            case RGB16:
                r = ( ( color >> 8 ) & 0xF8 ) | ( ( color & 0x8000 ) >> 13 );
                g = ( color >> 3 ) & 0xFC;
                b = ( ( color << 3 ) & 0xFC ) | ( ( color >> 3 ) & 0x01 );
                break;
            case RGB18:
                b = ( color << 2 ) & 0xFC;
                g = ( color >> 4 ) & 0xFC;
                r = ( color >> 10 ) & 0xFC;
                break;
            case RGB24:
                r = ( color >> 16 ) & 0xFC;
                g = ( color >> 8 ) & 0xFC;
                b = color & 0xFC;
                break;
        }
        WriteByteData( r );
        WriteByteData( g );
        WriteByteData( b );
    }
}

void HX8340S_LCD::serializeByte( unsigned char data )
{
    for ( int i = 0; i < 8; i++ ) {
        if ( data & 0x80 ) _lcd_pin_sdi = HIGH;
        else _lcd_pin_sdi = LOW;
        pulseLow( _lcd_pin_scl );
        data = data << 1;
    }
}
