#ifndef MBED_DMAFUNCGEN_H
#define MBED_DMAFUNCGEN_H

#include "mbed.h"
#include "MODDMA.h"

//! DMAFuncGen class generates a waveform on Analog output using Direct Memory Access alone.
class DMAFuncGen {
public:
    /// Create DMAFucGen instance
    /// @param dma The dma to use.
    /// @param channel The dma channel to use for transferring data to the D/A converter.
    DMAFuncGen(MODDMA& dma, MODDMA::CHANNELS channel);

    /// Number of data points in the waveform. Must be set by client.
    int buffer_size;
    /// Buffer for the binary data that will be transferred by DMA. Must be allocated by client.
    uint32_t* buffer;

    /// Set waveform value for a specified data point.
    /// @param idx The index of the data point in #buffer, must be in range 0..#buffer_size-1.
    /// @param x The value for the D/A, in the range 0..65535.
    void set(int idx, uint16_t x);
    /// Get waveform value for a specified data point
    /// @param idx The index of the data point in the #buffer, in the range 0..#buffer_size-1.
    /// @returns The value for the data point, in the range 0..65535.
    uint16_t operator[](const uint16_t idx);
    
    /// Connect the D/A converter pin (enable output)
    void Connect();
    /// Disconnect the D/A converter pin (pin becomes tri-state)
    void Disconnect();

    /// Reference to the DMA instance   
    MODDMA& dma;
    
    /// Set up the dma.
    /// Must set #buffer and #buffer_size first.
    void Setup(void);
    
    /// Set waveform frequency
    /// @note Must set #buffer_size first.
    /// @param f Frequency of a complete cycle of the waveform (in Hz). 
    void SetFrequency(float f);
    /// Read the actual waveform frequency that will be generated.
    float Frequency();
    
    /// Start generating the waveform (start dma)
    void Start();
    /// Stop generating the waveform (but waits for the current cycle to complete)
    void Stop();

private:
    MODDMA_Config conf;
    MODDMA_LLI lli;

    /// ISR routine, called on Terminal Count
    void TC_callback(void);
    /// ISR routine, called on dma ERRor
    void ERR_callback(void);
};

#endif
