/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//* LibTomCrypt, modular cryptographic library -- Tom St Denis
 *
 * LibTomCrypt is a library that provides various cryptographic
 * algorithms in a highly modular and flexible manner.
 *
 * The library is free for all purposes without any express
 * guarantee it works.
 *
 * Tom St Denis, tomstdenis@gmail.com, http://libtom.org
 */

/* Implements ECC over Z/pZ for curve y^2 = x^3 - 3x + b
 *
 * All curves taken from NIST recommendation paper of July 1999
 * Available at http://csrc.nist.gov/cryptval/dss.htm
 */
#include "ltc.h"

/**
  @file ltc_ecc_map.c
  ECC Crypto, Tom St Denis
*/  

#ifdef LTC_MECC

/**
  Map a projective jacbobian point back to affine space
  @param P        [in/out] The point to map
  @param modulus  The modulus of the field the ECC curve is in
  @param mp       The "b" value from montgomery_setup()
  @return MINITLS_OK on success
*/
int ltc_ecc_map(ecc_point *P, void *modulus, void *mp)
{
   fp_int t1, t2;
   int   err;

   LTC_ARGCHK(P       != NULL);
   LTC_ARGCHK(modulus != NULL);
   LTC_ARGCHK(mp      != NULL);

   if ((err = mp_init_multi(&t1, &t2, NULL)) != MINITLS_OK) {
      return MINITLS_ERR_MEMORY;
   }

   /* first map z back to normal */
   /*if ((err = */mp_montgomery_reduce(&P->z, modulus, mp);/*) != MINITLS_OK)           { goto done; }*/

   /* get 1/z */
   if ((err = mp_invmod(&P->z, modulus, &t1)) != MINITLS_OK)                      { goto done; }
 
   /* get 1/z^2 and 1/z^3 */
   /*if ((err = */mp_sqr(&t1, &t2);/*) != MINITLS_OK)                                    { goto done; }*/
   if ((err = mp_mod(&t2, modulus, &t2)) != MINITLS_OK)                           { goto done; }
   /*if ((err =*/ mp_mul(&t1, &t2, &t1);/*) != MINITLS_OK)                                { goto done; }*/
   /*if ((err =*/ mp_mod(&t1, modulus, &t1);/*) != MINITLS_OK)                           { goto done; }*/

   /* multiply against x/y */
   /*if ((err =*/ mp_mul(&P->x, &t2, &P->x);/*) != MINITLS_OK)                            { goto done; }*/
   /*if ((err =*/ mp_montgomery_reduce(&P->x, modulus, mp);/*) != MINITLS_OK)           { goto done; }*/
   /*if ((err =*/ mp_mul(&P->y, &t1, &P->y);/*) != MINITLS_OK)                            { goto done; }*/
   /*if ((err =*/ mp_montgomery_reduce(&P->y, modulus, mp);/*) != MINITLS_OK)           { goto done; }*/
   /*if ((err =*/ mp_set(&P->z, 1);/*) != MINITLS_OK)                                   { goto done; }*/

   err = MINITLS_OK;
done:
   mp_clear_multi(&t1, &t2, NULL);
   return err;
}

#endif

/* $Source: /cvs/libtom/libtomcrypt/src/pk/ecc/ltc_ecc_map.c,v $ */
/* $Revision: 1.7 $ */
/* $Date: 2007/05/12 14:32:35 $ */

