/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file crypto_hmac_sha256.c
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

#define __DEBUG__ 0
#ifndef __MODULE__
#define __MODULE__ "crypto_hmac_sha256.c"
#endif

#include "core/fwk.h"
#include "inc/minitls_errors.h"

#include "crypto_hmac_sha256.h"

void crypto_hmac_sha256_init(crypto_hmac_sha256_t* mac, const uint8_t* key, size_t size) //FIXME add crypto_err_t
{
  if(size > SHA256_BLOCK_SIZE)
  {
    //If key is longer than block size, hash the key first
    crypto_sha256_init(&mac->hash);
    crypto_sha256_update(&mac->hash, key, size);
    crypto_sha256_end(&mac->hash, mac->key_pad);
    size = SHA256_SIZE;
  }
  else
  {
    //Copy key
    memcpy(mac->key_pad, key, size);
  }

  //Pad key with 0s
  memset(mac->key_pad + size, 0, SHA256_BLOCK_SIZE - size);

  //XOR with 0x36363636...
  for( int i = 0; i < SHA256_BLOCK_SIZE / sizeof(uint32_t); i++)
  {
    //XOR by blocks
    *(((uint32_t*)mac->key_pad) + i) ^= 0x36363636UL;
  }

  //Init hash
  crypto_sha256_init(&mac->hash);
  crypto_sha256_update(&mac->hash, mac->key_pad, SHA256_BLOCK_SIZE);
}

void crypto_hmac_sha256_update(crypto_hmac_sha256_t* mac, const uint8_t* data, size_t size)
{
  crypto_sha256_update(&mac->hash, data, size);
}

void crypto_hmac_sha256_end(crypto_hmac_sha256_t* mac, uint8_t* out)
{
  //Complete hash
  uint8_t* hash1 = out; //Save memory
  crypto_sha256_end(&mac->hash, hash1);

  //Compute second key
  //XOR initial padded key with 0x5c5c5c5c...
  for( int i = 0; i < SHA256_BLOCK_SIZE / sizeof(uint32_t); i++)
  {
    //XOR by blocks
    *(((uint32_t*)mac->key_pad) + i) ^= (0x5c5c5c5cUL ^ 0x36363636UL);
  }

  //Compute final hash
  crypto_sha256_init(&mac->hash);
  crypto_sha256_update(&mac->hash, mac->key_pad, SHA256_BLOCK_SIZE);
  crypto_sha256_update(&mac->hash, hash1, SHA256_SIZE);
  crypto_sha256_end(&mac->hash, out);
}


