/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//* LibTomCrypt, modular cryptographic library -- Tom St Denis
 *
 * LibTomCrypt is a library that provides various cryptographic
 * algorithms in a highly modular and flexible manner.
 *
 * The library is free for all purposes without any express
 * guarantee it works.
 *
 * Tom St Denis, tomstdenis@gmail.com, http://libtom.org
 */

/* Implements ECC over Z/pZ for curve y^2 = x^3 - 3x + b
 *
 * All curves taken from NIST recommendation paper of July 1999
 * Available at http://csrc.nist.gov/cryptval/dss.htm
 */
#include "ltc.h"

/**
  @file ltc_ecc_mulmod_timing.c
  ECC Crypto, Tom St Denis
*/  

#ifdef LTC_MECC

#ifdef LTC_ECC_TIMING_RESISTANT

/**
   Perform a point multiplication  (timing resistant)
   @param k    The scalar to multiply by
   @param G    The base point
   @param R    [out] Destination for kG
   @param modulus  The modulus of the field the ECC curve is in
   @param map      Boolean whether to map back to affine or not (1==map, 0 == leave in projective)
   @return MINITLS_OK on success
*/
int ltc_ecc_mulmod(void *k, ecc_point *G, ecc_point *R, void *modulus, int map)
{
   ecc_point tG, M[3];
   int        i, j, err;
   fp_int     mu;
   fp_digit   mp;
   unsigned long buf;
   int        first, bitbuf, bitcpy, bitcnt, mode, digidx;

   LTC_ARGCHK(k       != NULL);
   LTC_ARGCHK(G       != NULL);
   LTC_ARGCHK(R       != NULL);
   LTC_ARGCHK(modulus != NULL);

   /* init montgomery reduction */
   if ((err = mp_montgomery_setup(modulus, &mp)) != MINITLS_OK) {
      return err;
   }
   /*if ((err =*/ mp_init(&mu);/*) != MINITLS_OK) {
      mp_montgomery_free(&mp);
      return err;
   }*/
   /*if ((err =*/ mp_montgomery_normalization(&mu, modulus);/*) != MINITLS_OK) {
      mp_clear(&mu);
      mp_montgomery_free(&mp);
      return err;
   }*/

  /* alloc ram for window temps */
  for (i = 0; i < 3; i++) {
    if (mp_init_multi(&M[i].x, &M[i].y, &M[i].z, NULL) != MINITLS_OK)
    {
         for (j = 0; j < i; j++) {
           mp_clear_multi(&M[j].x, &M[j].y, &M[j].z, NULL);
         }
         mp_clear(&mu);
         mp_montgomery_free(&mp);
         return MINITLS_ERR_MEMORY;
      }
  }

   /* make a copy of G incase R==G */
   if (mp_init_multi(&tG.x, &tG.y, &tG.z, NULL) != MINITLS_OK)                      { err = MINITLS_ERR_MEMORY; goto done; }

   /* tG = G  and convert to montgomery */
   if ((err = mp_mulmod(&G->x, &mu, modulus, &tG.x)) != MINITLS_OK)                      { goto done; }
   if ((err = mp_mulmod(&G->y, &mu, modulus, &tG.y)) != MINITLS_OK)                      { goto done; }
   if ((err = mp_mulmod(&G->z, &mu, modulus, &tG.z)) != MINITLS_OK)                      { goto done; }
   mp_clear(&mu);
   //mu = NULL;
   
   /* calc the M tab */
   /* M[0] == G */
   /*if ((err =*/ mp_copy(&tG.x, &M[0].x); /*) != MINITLS_OK)                                  { goto done; }*/
   /*if ((err =*/ mp_copy(&tG.y, &M[0].y); /*) != MINITLS_OK)                                  { goto done; }*/
   /*if ((err =*/ mp_copy(&tG.z, &M[0].z); /*) != MINITLS_OK)                                  { goto done; }*/
   /* M[1] == 2G */
   if ((err = ltc_ecc_projective_dbl_point(&tG, &M[1], modulus, &mp)) != MINITLS_OK)                  { goto done; }

   /* setup sliding window */
   mode   = 0;
   bitcnt = 1;
   buf    = 0;
   digidx = mp_get_digit_count(k) - 1;
   bitcpy = bitbuf = 0;
   first  = 1;

   /* perform ops */
   for (;;) {
     /* grab next digit as required */
      if (--bitcnt == 0) {
         if (digidx == -1) {
            break;
         }
         buf    = mp_get_digit(k, digidx);
         bitcnt = (int) MP_DIGIT_BIT;
         --digidx;
      }

      /* grab the next msb from the ltiplicand */
      i = (buf >> (MP_DIGIT_BIT - 1)) & 1;
      buf <<= 1;

      if (mode == 0 && i == 0) {
         /* dummy operations */
         if ((err = ltc_ecc_projective_add_point(&M[0], &M[1], &M[2], modulus, &mp)) != MINITLS_OK)    { goto done; }
         if ((err = ltc_ecc_projective_dbl_point(&M[1], &M[2], modulus, &mp)) != MINITLS_OK)          { goto done; }
         continue;
      }

      if (mode == 0 && i == 1) {
         mode = 1;
         /* dummy operations */
         if ((err = ltc_ecc_projective_add_point(&M[0], &M[1], &M[2], modulus, &mp)) != MINITLS_OK)    { goto done; }
         if ((err = ltc_ecc_projective_dbl_point(&M[1], &M[2], modulus, &mp)) != MINITLS_OK)          { goto done; }
         continue;
      }

      if ((err = ltc_ecc_projective_add_point(&M[0], &M[1], &M[i^1], modulus, &mp)) != MINITLS_OK)     { goto done; }
      if ((err = ltc_ecc_projective_dbl_point(&M[i], &M[i], modulus, &mp)) != MINITLS_OK)             { goto done; }
   }

   /* copy result out */
   /*if ((err =*/ mp_copy(&M[0].x, &R->x);/*) != MINITLS_OK)                                   { goto done; }*/
   /*if ((err =*/ mp_copy(&M[0].y, &R->y);/*) != MINITLS_OK)                                   { goto done; }*/
   /*if ((err =*/ mp_copy(&M[0].z, &R->z);/*) != MINITLS_OK)                                   { goto done; }*/

   /* map R back from projective space */
   if (map) {
      err = ltc_ecc_map(R, modulus, &mp);
   } else {
      err = MINITLS_OK;
   }
done:
   //if (mu != NULL) {
      mp_clear(&mu);
   //}
   mp_montgomery_free(&mp);
   mp_clear_multi(&tG.x, &tG.y, &tG.z, NULL);
   for (i = 0; i < 3; i++) {
     mp_clear_multi(&M[i].x, &M[i].y, &M[i].z, NULL);
   }
   return err;
}

#endif
#endif
/* $Source: /cvs/libtom/libtomcrypt/src/pk/ecc/ltc_ecc_mulmod_timing.c,v $ */
/* $Revision: 1.13 $ */
/* $Date: 2007/05/12 14:32:35 $ */

