/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file crypto_sha256.c
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

#define __DEBUG__ 0
#define __MODULE__ "crypto_sha256.c"

//This module has been adapted from libtomcrypt (http://libtom.org/)

#include "core/fwk.h"
#include "crypto_sha256.h"
#include "inc/minitls_errors.h"
#include "crypto_macros.h"

#define fatal(x) do{ ERR("Fatal error %s - %d", #x, x); while(1); }while(0)

static void crypto_sha256_compress(crypto_sha256_t* hash, unsigned char *buf);

void crypto_sha256_init(crypto_sha256_t* hash)
{
  hash->state[0] = 0x6A09E667UL;
  hash->state[1] = 0xBB67AE85UL;
  hash->state[2] = 0x3C6EF372UL;
  hash->state[3] = 0xA54FF53AUL;
  hash->state[4] = 0x510E527FUL;
  hash->state[5] = 0x9B05688CUL;
  hash->state[6] = 0x1F83D9ABUL;
  hash->state[7] = 0x5BE0CD19UL;
  hash->curlen = 0;
  hash->length = 0;
}

void crypto_sha256_update(crypto_sha256_t* hash, const uint8_t* data, size_t size)
{
  unsigned long n;

  if ( hash->curlen > sizeof( hash->buf)) {
     //return CRYPTO_ERR_PARAMETERS;
    fatal(CRYPTO_ERR_PARAMETERS);
  }
  while (size > 0) {
      if ( hash->curlen == 0 && size >= 64) {
         crypto_sha256_compress(hash, (unsigned char *)data);
         hash->length += 64 * 8;
         data             += 64;
         size          -= 64;
      } else {
         n = ( ((size)<((64 -  hash->curlen)))?(size):((64 -  hash->curlen)) );
         memcpy( hash->buf + hash->curlen, data, (size_t)n);
         hash->curlen += n;
         data         += n;
         size         -= n;
         if ( hash->curlen == 64) {
            crypto_sha256_compress (hash,  hash->buf);
             hash->length += 8*64;
             hash->curlen = 0;
         }
     }
  }
}

void crypto_sha256_end(crypto_sha256_t* hash, uint8_t* out)
{
  int i;

  if (hash->curlen >= sizeof(hash->buf)) {
    fatal(CRYPTO_ERR_PARAMETERS);
  }

  /* increase the length of the message */
  hash->length += hash->curlen * 8;

  /* append the '1' bit */
  hash->buf[hash->curlen++] = (unsigned char)0x80;

  /* if the length is currently above 56 bytes we append zeros
   * then compress.  Then we can fall back to padding zeros and length
   * encoding like normal.
   */
  if (hash->curlen > 56) {
      while (hash->curlen < 64) {
          hash->buf[hash->curlen++] = (unsigned char)0;
      }
      crypto_sha256_compress(hash, hash->buf);
      hash->curlen = 0;
  }

  /* pad upto 56 bytes of zeroes */
  while (hash->curlen < 56) {
      hash->buf[hash->curlen++] = (unsigned char)0;
  }

  /* store length */
  STORE64H(hash->length, hash->buf+56);
  crypto_sha256_compress(hash, hash->buf);

  /* copy output */
  for (i = 0; i < 8; i++) {
      STORE32H(hash->state[i], out+(4*i));
  }
  #ifdef CRYPT_CLEAN_STACK
    zeromem(hash, sizeof(hash));
  #endif
}

void crypto_sha256_copy(crypto_sha256_t* hashTo, crypto_sha256_t* hashFrom)
{
  memcpy(hashTo, hashFrom, sizeof(crypto_sha256_t));
}

/* Various logical functions */
#define Ch(x,y,z)       (z ^ (x & (y ^ z)))
#define Maj(x,y,z)      (((x | y) & z) | (x & y))
#define S(x, n)         RORc((x),(n))
#define R(x, n)         (((x)&0xFFFFFFFFUL)>>(n))
#define Sigma0(x)       (S(x, 2) ^ S(x, 13) ^ S(x, 22))
#define Sigma1(x)       (S(x, 6) ^ S(x, 11) ^ S(x, 25))
#define Gamma0(x)       (S(x, 7) ^ S(x, 18) ^ R(x, 3))
#define Gamma1(x)       (S(x, 17) ^ S(x, 19) ^ R(x, 10))

void crypto_sha256_compress(crypto_sha256_t* hash, unsigned char *buf)
{
  ulong32 S[8], W[64], t0, t1;
#ifdef LTC_SMALL_CODE
  ulong32 t;
#endif
  int i;

  /* copy state into S */
  for (i = 0; i < 8; i++) {
      S[i] = hash->state[i];
  }

  /* copy the state into 512-bits into W[0..15] */
  for (i = 0; i < 16; i++) {
      LOAD32H(W[i], buf + (4*i));
  }

  /* fill W[16..63] */
  for (i = 16; i < 64; i++) {
      W[i] = Gamma1(W[i - 2]) + W[i - 7] + Gamma0(W[i - 15]) + W[i - 16];
  }

  /* Compress */
#ifdef LTC_SMALL_CODE
#define RND(a,b,c,d,e,f,g,h,i)                         \
   t0 = h + Sigma1(e) + Ch(e, f, g) + K[i] + W[i];   \
   t1 = Sigma0(a) + Maj(a, b, c);                    \
   d += t0;                                          \
   h  = t0 + t1;

   for (i = 0; i < 64; ++i) {
       RND(S[0],S[1],S[2],S[3],S[4],S[5],S[6],S[7],i);
       t = S[7]; S[7] = S[6]; S[6] = S[5]; S[5] = S[4];
       S[4] = S[3]; S[3] = S[2]; S[2] = S[1]; S[1] = S[0]; S[0] = t;
   }
#else
#define RND(a,b,c,d,e,f,g,h,i,ki)                    \
   t0 = h + Sigma1(e) + Ch(e, f, g) + ki + W[i];   \
   t1 = Sigma0(a) + Maj(a, b, c);                  \
   d += t0;                                        \
   h  = t0 + t1;

  RND(S[0],S[1],S[2],S[3],S[4],S[5],S[6],S[7],0,0x428a2f98);
  RND(S[7],S[0],S[1],S[2],S[3],S[4],S[5],S[6],1,0x71374491);
  RND(S[6],S[7],S[0],S[1],S[2],S[3],S[4],S[5],2,0xb5c0fbcf);
  RND(S[5],S[6],S[7],S[0],S[1],S[2],S[3],S[4],3,0xe9b5dba5);
  RND(S[4],S[5],S[6],S[7],S[0],S[1],S[2],S[3],4,0x3956c25b);
  RND(S[3],S[4],S[5],S[6],S[7],S[0],S[1],S[2],5,0x59f111f1);
  RND(S[2],S[3],S[4],S[5],S[6],S[7],S[0],S[1],6,0x923f82a4);
  RND(S[1],S[2],S[3],S[4],S[5],S[6],S[7],S[0],7,0xab1c5ed5);
  RND(S[0],S[1],S[2],S[3],S[4],S[5],S[6],S[7],8,0xd807aa98);
  RND(S[7],S[0],S[1],S[2],S[3],S[4],S[5],S[6],9,0x12835b01);
  RND(S[6],S[7],S[0],S[1],S[2],S[3],S[4],S[5],10,0x243185be);
  RND(S[5],S[6],S[7],S[0],S[1],S[2],S[3],S[4],11,0x550c7dc3);
  RND(S[4],S[5],S[6],S[7],S[0],S[1],S[2],S[3],12,0x72be5d74);
  RND(S[3],S[4],S[5],S[6],S[7],S[0],S[1],S[2],13,0x80deb1fe);
  RND(S[2],S[3],S[4],S[5],S[6],S[7],S[0],S[1],14,0x9bdc06a7);
  RND(S[1],S[2],S[3],S[4],S[5],S[6],S[7],S[0],15,0xc19bf174);
  RND(S[0],S[1],S[2],S[3],S[4],S[5],S[6],S[7],16,0xe49b69c1);
  RND(S[7],S[0],S[1],S[2],S[3],S[4],S[5],S[6],17,0xefbe4786);
  RND(S[6],S[7],S[0],S[1],S[2],S[3],S[4],S[5],18,0x0fc19dc6);
  RND(S[5],S[6],S[7],S[0],S[1],S[2],S[3],S[4],19,0x240ca1cc);
  RND(S[4],S[5],S[6],S[7],S[0],S[1],S[2],S[3],20,0x2de92c6f);
  RND(S[3],S[4],S[5],S[6],S[7],S[0],S[1],S[2],21,0x4a7484aa);
  RND(S[2],S[3],S[4],S[5],S[6],S[7],S[0],S[1],22,0x5cb0a9dc);
  RND(S[1],S[2],S[3],S[4],S[5],S[6],S[7],S[0],23,0x76f988da);
  RND(S[0],S[1],S[2],S[3],S[4],S[5],S[6],S[7],24,0x983e5152);
  RND(S[7],S[0],S[1],S[2],S[3],S[4],S[5],S[6],25,0xa831c66d);
  RND(S[6],S[7],S[0],S[1],S[2],S[3],S[4],S[5],26,0xb00327c8);
  RND(S[5],S[6],S[7],S[0],S[1],S[2],S[3],S[4],27,0xbf597fc7);
  RND(S[4],S[5],S[6],S[7],S[0],S[1],S[2],S[3],28,0xc6e00bf3);
  RND(S[3],S[4],S[5],S[6],S[7],S[0],S[1],S[2],29,0xd5a79147);
  RND(S[2],S[3],S[4],S[5],S[6],S[7],S[0],S[1],30,0x06ca6351);
  RND(S[1],S[2],S[3],S[4],S[5],S[6],S[7],S[0],31,0x14292967);
  RND(S[0],S[1],S[2],S[3],S[4],S[5],S[6],S[7],32,0x27b70a85);
  RND(S[7],S[0],S[1],S[2],S[3],S[4],S[5],S[6],33,0x2e1b2138);
  RND(S[6],S[7],S[0],S[1],S[2],S[3],S[4],S[5],34,0x4d2c6dfc);
  RND(S[5],S[6],S[7],S[0],S[1],S[2],S[3],S[4],35,0x53380d13);
  RND(S[4],S[5],S[6],S[7],S[0],S[1],S[2],S[3],36,0x650a7354);
  RND(S[3],S[4],S[5],S[6],S[7],S[0],S[1],S[2],37,0x766a0abb);
  RND(S[2],S[3],S[4],S[5],S[6],S[7],S[0],S[1],38,0x81c2c92e);
  RND(S[1],S[2],S[3],S[4],S[5],S[6],S[7],S[0],39,0x92722c85);
  RND(S[0],S[1],S[2],S[3],S[4],S[5],S[6],S[7],40,0xa2bfe8a1);
  RND(S[7],S[0],S[1],S[2],S[3],S[4],S[5],S[6],41,0xa81a664b);
  RND(S[6],S[7],S[0],S[1],S[2],S[3],S[4],S[5],42,0xc24b8b70);
  RND(S[5],S[6],S[7],S[0],S[1],S[2],S[3],S[4],43,0xc76c51a3);
  RND(S[4],S[5],S[6],S[7],S[0],S[1],S[2],S[3],44,0xd192e819);
  RND(S[3],S[4],S[5],S[6],S[7],S[0],S[1],S[2],45,0xd6990624);
  RND(S[2],S[3],S[4],S[5],S[6],S[7],S[0],S[1],46,0xf40e3585);
  RND(S[1],S[2],S[3],S[4],S[5],S[6],S[7],S[0],47,0x106aa070);
  RND(S[0],S[1],S[2],S[3],S[4],S[5],S[6],S[7],48,0x19a4c116);
  RND(S[7],S[0],S[1],S[2],S[3],S[4],S[5],S[6],49,0x1e376c08);
  RND(S[6],S[7],S[0],S[1],S[2],S[3],S[4],S[5],50,0x2748774c);
  RND(S[5],S[6],S[7],S[0],S[1],S[2],S[3],S[4],51,0x34b0bcb5);
  RND(S[4],S[5],S[6],S[7],S[0],S[1],S[2],S[3],52,0x391c0cb3);
  RND(S[3],S[4],S[5],S[6],S[7],S[0],S[1],S[2],53,0x4ed8aa4a);
  RND(S[2],S[3],S[4],S[5],S[6],S[7],S[0],S[1],54,0x5b9cca4f);
  RND(S[1],S[2],S[3],S[4],S[5],S[6],S[7],S[0],55,0x682e6ff3);
  RND(S[0],S[1],S[2],S[3],S[4],S[5],S[6],S[7],56,0x748f82ee);
  RND(S[7],S[0],S[1],S[2],S[3],S[4],S[5],S[6],57,0x78a5636f);
  RND(S[6],S[7],S[0],S[1],S[2],S[3],S[4],S[5],58,0x84c87814);
  RND(S[5],S[6],S[7],S[0],S[1],S[2],S[3],S[4],59,0x8cc70208);
  RND(S[4],S[5],S[6],S[7],S[0],S[1],S[2],S[3],60,0x90befffa);
  RND(S[3],S[4],S[5],S[6],S[7],S[0],S[1],S[2],61,0xa4506ceb);
  RND(S[2],S[3],S[4],S[5],S[6],S[7],S[0],S[1],62,0xbef9a3f7);
  RND(S[1],S[2],S[3],S[4],S[5],S[6],S[7],S[0],63,0xc67178f2);

#undef RND

#endif

  /* feedback */
  for (i = 0; i < 8; i++) {
      hash->state[i] = hash->state[i] + S[i];
  }

#if CRYPT_CLEAN_STACK
  burn_stack(sizeof(ulong32) * 74);
#endif
}
