/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file tls_record.h
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

#ifndef TLS_RECORD_H_
#define TLS_RECORD_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "core/fwk.h"
#include "inc/minitls_errors.h"

#define TLS_DEFAULT_MAX_FRAGMENT_SIZE 18432 //(MAX 2^14 + 2048 = 18432) -- encrypted

#define TLS_ENCRYPTION_MAX_OVERHEAD (20 + 256 + 16) //MAC + Max Padding + IV

#include "tls_socket_defs.h"

/*
 * When a new session
   begins, the record layer's connection state encryption, hash, and
   compression algorithms are initialized to null.  The current
   connection state is used for renegotiation messages.
 */
minitls_err_t tls_record_init(tls_record_t* record, tls_socket_t* socket, uint8_t* buf, size_t buf_size);

//Should be called after server hello message;
void tls_record_set_protocol_version(tls_record_t* record, uint8_t major, uint8_t minor);
void tls_record_get_protocol_version(tls_record_t* record, uint8_t* major, uint8_t* minor);

minitls_err_t tls_record_change_cipher_spec(tls_record_t* record, bool tx_nrx);

bool tls_record_is_secure(tls_record_t* record);

//Read on message and process it
//TODO mutex this
minitls_err_t tls_record_connect(tls_record_t* record, const char* hostname, uint16_t port);

minitls_err_t tls_record_process(tls_record_t* record);
minitls_err_t tls_record_send(tls_record_t* record, tls_content_type_t content_type, buffer_t* payload);

//Keys will be copied in local buffer
minitls_err_t tls_record_set_keys(tls_record_t* record, tls_security_type_t security, const uint8_t* client_write_mac_key,
    const uint8_t* server_write_mac_key, const uint8_t* client_write_cipher_key, const uint8_t* server_write_cipher_key);

minitls_err_t tls_record_close(tls_record_t* record);

minitls_err_t tls_record_set_read_timeout(tls_record_t* record, int timeout);
minitls_err_t tls_record_set_write_timeout(tls_record_t* record, int timeout);

#ifdef __cplusplus
}
#endif

#endif /* TLS_RECORD_H_ */
