/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file tls_alert.c
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

#define __DEBUG__ 4
#ifndef __MODULE__
#define __MODULE__ "tls_alert.c"
#endif

#include "core/fwk.h"
#include "inc/minitls_errors.h"
#include "tls_alert.h"

minitls_err_t tls_alert_send( tls_record_t* record, tls_message_alert_level_t level, tls_message_alert_description_t description, buffer_t* buffer_tx )
{
  if( buffer_size(buffer_tx) < 2 )
  {
    return MINITLS_ERR_BUFFER_TOO_SMALL;
  }

  buffer_reset(buffer_tx);

  buffer_nu8_write(buffer_tx, level);
  buffer_nu8_write(buffer_tx, description);

  minitls_err_t ret = tls_record_send(record, TLS_ALERT, buffer_tx);
  if(ret)
  {
    return ret;
  }

  return MINITLS_OK;
}

minitls_err_t tls_alert_process( tls_record_t* record, buffer_t* buffer_rx )
{
  if( buffer_length(buffer_rx) < 2 )
  {
    return MINITLS_ERR_BUFFER_TOO_SMALL;
  }

  tls_message_alert_level_t level = buffer_nu8_read(buffer_rx);
  tls_message_alert_description_t description = buffer_nu8_read(buffer_rx);

  WARN("Alert: type: %d, description: %d", level, description);
  if(description == CLOSE_NOTIFY)
  {
    return MINITLS_ERR_CONNECTION_CLOSED;
  }

  //Always return error
  return MINITLS_ERR_PEER;
}
