/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file socket_lwip.c
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

#define __DEBUG__ 4
#ifndef __MODULE__
#define __MODULE__ "socket_lwip.c"
#endif

#include "core/fwk.h"

#include "socket.h"

#include "lwip/sockets.h"
#include "lwip/inet.h"
#include "lwip/netdb.h"

int socket_socket()
{
  return lwip_socket(AF_INET, SOCK_STREAM, 0);
}


int socket_connect(int fd, const char* hostname, uint16_t port)
{
  struct hostent* hostent = lwip_gethostbyname(hostname);
  if(hostent == NULL)
  { 
    WARN("Could not resolve %s\r\n", hostname);
    return -1;
  }
  struct sockaddr_in server;
  memset(&server, 0, sizeof(struct sockaddr_in));
  server.sin_family = AF_INET;
  memcpy(&server.sin_addr.s_addr, hostent->h_addr_list[0], hostent->h_length);
  //server.sin_addr.s_addr = inet_addr(hostname);
  server.sin_port = htons(port);

  DBG("Trying to connect to %s:%d", inet_ntoa(server.sin_addr.s_addr), port);

  return lwip_connect(fd, (const struct sockaddr*)&server, sizeof(struct sockaddr_in));
}

int socket_wait_readable(int fd, int timeout_ms)
{
  //Wait for record to be readable
  fd_set set;
  FD_ZERO(&set);
  FD_SET(fd, &set);

  struct timeval timeout;
  timeout.tv_sec = timeout_ms / 1000;
  timeout.tv_usec = (timeout_ms - (timeout.tv_sec * 1000)) * 1000;

  int r = lwip_select(fd + 1, &set, NULL, NULL, &timeout);
  if( (r <= 0) || !FD_ISSET(fd, &set) )
  {
    return -1;
  }
  return 0;
}

int socket_wait_writeable(int fd, int timeout_ms)
{
  //Wait for record to be writeable
  fd_set set;
  FD_ZERO(&set);
  FD_SET(fd, &set);

  struct timeval timeout;
  timeout.tv_sec = timeout_ms / 1000;
  timeout.tv_usec = (timeout_ms - (timeout.tv_sec * 1000)) * 1000;

  int r = lwip_select(fd + 1, NULL, &set, NULL, &timeout);
  if( (r <= 0) || !FD_ISSET(fd, &set) )
  {
    return -1;
  }
  return 0;
}

int socket_recv(int fd, void* buf, size_t size)
{
  return lwip_recv(fd, buf, size, 0);
}

int socket_send(int fd, void* buf, size_t size)
{
  return lwip_send(fd, buf, size, 0);
}

int socket_close(int fd)
{
  return lwip_close(fd);
}
