/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//* LibTomCrypt, modular cryptographic library -- Tom St Denis
 *
 * LibTomCrypt is a library that provides various cryptographic
 * algorithms in a highly modular and flexible manner.
 *
 * The library is free for all purposes without any express
 * guarantee it works.
 *
 * Tom St Denis, tomstdenis@gmail.com, http://libtom.org
 */

/* Implements ECC over Z/pZ for curve y^2 = x^3 - 3x + b
 *
 * All curves taken from NIST recommendation paper of July 1999
 * Available at http://csrc.nist.gov/cryptval/dss.htm
 */
#include "ltc.h"

/**
  @file ltc_ecc_projective_dbl_point.c
  ECC Crypto, Tom St Denis
*/  

#if defined(LTC_MECC) && (!defined(LTC_MECC_ACCEL) || defined(LTM_LTC_DESC))

/**
   Double an ECC point
   @param P   The point to double
   @param R   [out] The destination of the double
   @param modulus  The modulus of the field the ECC curve is in
   @param mp       The "b" value from montgomery_setup()
   @return MINITLS_OK on success
*/
int ltc_ecc_projective_dbl_point(ecc_point *P, ecc_point *R, void *modulus, void *mp)
{
   fp_int t1, t2;
   int   err;

   LTC_ARGCHK(P       != NULL);
   LTC_ARGCHK(R       != NULL);
   LTC_ARGCHK(modulus != NULL);
   LTC_ARGCHK(mp      != NULL);

   if ((err = mp_init_multi(&t1, &t2, NULL)) != MINITLS_OK){
      return err;
   }

   if (P != R) {
      /*if ((err = */mp_copy(&P->x, &R->x);/*) != MINITLS_OK)                             { goto done; }*/
      /*if ((err = */mp_copy(&P->y, &R->y);/*) != MINITLS_OK)                             { goto done; }*/
      /*if ((err = */mp_copy(&P->z, &R->z);/*) != MINITLS_OK)                             { goto done; }*/
   }

   /* &t1 = Z * Z */
   /*if ((err = */mp_sqr(&R->z, &t1);/*) != MINITLS_OK)                                   { goto done; }*/
   /*if ((err = */mp_montgomery_reduce(&t1, modulus, mp);/*) != MINITLS_OK)                { goto done; }*/
   /* Z = Y * Z */
   /*if ((err = */mp_mul(&R->z, &R->y, &R->z);/*) != MINITLS_OK)                             { goto done; }*/
   /*if ((err = */mp_montgomery_reduce(&R->z, modulus, mp);/*) != MINITLS_OK)            { goto done; }*/
   /* Z = 2Z */
   /*if ((err = */mp_add(&R->z, &R->z, &R->z);/*) != MINITLS_OK)                             { goto done; }*/
   if (mp_cmp(&R->z, modulus) != MP_LT) {
      /*if ((err = */mp_sub(&R->z, modulus, &R->z);/*) != MINITLS_OK)                     { goto done; }*/
   }
   
   /* T2 = X - T1 */
   /*if ((err = */mp_sub(&R->x, &t1, &t2);/*) != MINITLS_OK)                                 { goto done; }*/
   if (mp_cmp_d(&t2, 0) == MP_LT) {
      /*if ((err = */mp_add(&t2, modulus, &t2);/*) != MINITLS_OK)                           { goto done; }*/
   }
   /* T1 = X + T1 */
   /*if ((err = */mp_add(&t1, &R->x, &t1);/*) != MINITLS_OK)                                 { goto done; }*/
   if (mp_cmp(&t1, modulus) != MP_LT) {
      /*if ((err = */mp_sub(&t1, modulus, &t1);/*) != MINITLS_OK)                           { goto done; }*/
   }
   /* T2 = T1 * T2 */
   /*if ((err = */mp_mul(&t1, &t2, &t2);/*) != MINITLS_OK)                                   { goto done; }*/
   /*if ((err = */mp_montgomery_reduce(&t2, modulus, mp);/*) != MINITLS_OK)                { goto done; }*/
   /* T1 = 2T2 */
   /*if ((err = */mp_add(&t2, &t2, &t1);/*) != MINITLS_OK)                                   { goto done; }*/
   if (mp_cmp(&t1, modulus) != MP_LT) {
      /*if ((err = */mp_sub(&t1, modulus, &t1);/*) != MINITLS_OK)                           { goto done; }*/
   }
   /* T1 = T1 + T2 */
   /*if ((err = */mp_add(&t1, &t2, &t1);/*) != MINITLS_OK)                                   { goto done; }*/
   if (mp_cmp(&t1, modulus) != MP_LT) {
      /*if ((err = */mp_sub(&t1, modulus, &t1);/*) != MINITLS_OK)                           { goto done; }*/
   }

   /* Y = 2Y */
   /*if ((err = */mp_add(&R->y, &R->y, &R->y);/*) != MINITLS_OK)                             { goto done; }*/
   if (mp_cmp(&R->y, modulus) != MP_LT) {
      /*if ((err = */mp_sub(&R->y, modulus, &R->y);/*) != MINITLS_OK)                     { goto done; }*/
   }
   /* Y = Y * Y */
   /*if ((err = */mp_sqr(&R->y, &R->y);/*) != MINITLS_OK)                                   { goto done; }*/
   /*if ((err = */mp_montgomery_reduce(&R->y, modulus, mp);/*) != MINITLS_OK)            { goto done; }*/
   /* T2 = Y * Y */
   /*if ((err = */mp_sqr(&R->y, &t2);/*) != MINITLS_OK)                                   { goto done; }*/
   /*if ((err = */mp_montgomery_reduce(&t2, modulus, mp);/*) != MINITLS_OK)                { goto done; }*/
   /* T2 = T2/2 */
   if (mp_isodd(&t2)) {
      /*if ((err = */mp_add(&t2, modulus, &t2);/*) != MINITLS_OK)                           { goto done; }*/
   }
   /*if ((err = */mp_div_2(&t2, &t2);/*) != MINITLS_OK)                                   { goto done; }*/
   /* Y = Y * X */
   /*if ((err = */mp_mul(&R->y, &R->x, &R->y);/*) != MINITLS_OK)                             { goto done; }*/
   /*if ((err = */mp_montgomery_reduce(&R->y, modulus, mp);/*) != MINITLS_OK)            { goto done; }*/

   /* X  = T1 * T1 */
   /*if ((err = */mp_sqr(&t1, &R->x);/*) != MINITLS_OK)                                   { goto done; }*/
   /*if ((err = */mp_montgomery_reduce(&R->x, modulus, mp);/*) != MINITLS_OK)            { goto done; }*/
   /* X = X - Y */
   /*if ((err = */mp_sub(&R->x, &R->y, &R->x);/*) != MINITLS_OK)                             { goto done; }*/
   if (mp_cmp_d(&R->x, 0) == MP_LT) {
      /*if ((err = */mp_add(&R->x, modulus, &R->x);/*) != MINITLS_OK)                     { goto done; }*/
   }
   /* X = X - Y */
   /*if ((err = */mp_sub(&R->x, &R->y, &R->x);/*) != MINITLS_OK)                             { goto done; }*/
   if (mp_cmp_d(&R->x, 0) == MP_LT) {
      /*if ((err = */mp_add(&R->x, modulus, &R->x);/*) != MINITLS_OK)                     { goto done; }*/
   }

   /* Y = Y - X */     
   /*if ((err = */mp_sub(&R->y, &R->x, &R->y);/*) != MINITLS_OK)                             { goto done; }*/
   if (mp_cmp_d(&R->y, 0) == MP_LT) {
      /*if ((err = */mp_add(&R->y, modulus, &R->y);/*) != MINITLS_OK)                     { goto done; }*/
   }
   /* Y = Y * T1 */
   /*if ((err = */mp_mul(&R->y, &t1, &R->y);/*) != MINITLS_OK)                             { goto done; }*/
   /*if ((err = */mp_montgomery_reduce(&R->y, modulus, mp);/*) != MINITLS_OK)            { goto done; }*/
   /* Y = Y - T2 */
   /*if ((err = */mp_sub(&R->y, &t2, &R->y);/*) != MINITLS_OK)                             { goto done; }*/
   if (mp_cmp_d(&R->y, 0) == MP_LT) {
      /*if ((err = */mp_add(&R->y, modulus, &R->y);/*) != MINITLS_OK)                     { goto done; }*/
   }
 
   err = MINITLS_OK;
/*done:*/ //Unused
   mp_clear_multi(&t1, &t2, NULL);
   return err;
}
#endif
/* $Source: /cvs/libtom/libtomcrypt/src/pk/ecc/ltc_ecc_projective_dbl_point.c,v $ */
/* $Revision: 1.11 $ */
/* $Date: 2007/05/12 14:32:35 $ */

