/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file crypto_prng.c
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

#define __DEBUG__ 0//4
#ifndef __MODULE__
#define __MODULE__ "crypto_prng.c"
#endif

#include "core/fwk.h"
#include "inc/minitls_errors.h"
#include "crypto_prng.h"

#include "crypto_aes_128.h"
#include "crypto_sha1.h"

#define YARROW_REGENERATE_VALUE 10

static void crypto_prng_update_internal(crypto_prng_t* prng);

void crypto_prng_init(crypto_prng_t* prng, rtos_mtx_t* mtx)
{
  memset(prng->pool, 0, SHA1_SIZE);
  prng->counter = 0xFFFFFFFFUL; //Will force update
  prng->mtx = mtx;
  prng->buf_pos = AES_128_BLOCK_SIZE; //invalidate buffer
  prng->fed = false;
}

#define LOCK() do{ if(prng->mtx) { rtos_mtx_lock(prng->mtx); } }while(0)
#define UNLOCK() do{ if(prng->mtx) { rtos_mtx_unlock(prng->mtx); } }while(0)

void crypto_prng_feed(crypto_prng_t* prng, uint8_t* data, size_t size)
{
  LOCK();
  //Update pool
  crypto_sha1_t hash;
  crypto_sha1_init(&hash);
  crypto_sha1_update(&hash, prng->pool, SHA1_SIZE);
  crypto_sha1_update(&hash, data, size);
  crypto_sha1_end(&hash, prng->pool);
  prng->fed = true;
  UNLOCK();
}

void crypto_prng_update_internal(crypto_prng_t* prng)
{
  if(!prng->fed)
  {
    //Rehash the pool to "rotate" key
    crypto_sha1_t hash;
    crypto_sha1_init(&hash);
    crypto_sha1_update(&hash, prng->pool, SHA1_SIZE);
    crypto_sha1_end(&hash, prng->pool);
  }
  prng->fed = false;
  //Generate key from pool
  crypto_aes_128_init(&prng->cipher, prng->pool, expand_encryption_key); //OK because AES_128_BLOCK_SIZE < SHA1_SIZE
  prng->counter = 0;
}

void crypto_prng_update(crypto_prng_t* prng)
{
  LOCK();
  crypto_prng_update_internal(prng);
  UNLOCK();
}

void crypto_prng_get(crypto_prng_t* prng, uint8_t* data, size_t size)
{
  LOCK();

  //Regenerate data as needed
  while(size > 0)
  {
    if(prng->counter > YARROW_REGENERATE_VALUE)
    {
      crypto_prng_update_internal(prng);
    }

    //Copy any remaining data from buffer
    size_t cpy_size = MIN(size, AES_128_BLOCK_SIZE - prng->buf_pos);
    memcpy(data, prng->buf + prng->buf_pos, cpy_size);
    data += cpy_size;
    size -= cpy_size;
    prng->buf_pos += cpy_size;

    if(prng->buf_pos >= AES_128_BLOCK_SIZE)
    {
      memset(prng->buf + sizeof(uint32_t), 0, AES_128_BLOCK_SIZE - sizeof(uint32_t));
      memcpy(prng->buf, &prng->counter, sizeof(uint32_t)); //We do not care about endianness as long as it's consistent (don't know of any system which would swap endianness during the execution of a program...)
      crypto_aes_128_encrypt(&prng->cipher, prng->buf, prng->buf);
      prng->counter++;
      prng->buf_pos = 0;
    }
  }

  UNLOCK();
}


