/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file crypto_ecc.h
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

//This module has been adapted from libtomcrypt (http://libtom.org/)

#ifndef CRYPTO_ECC_H_
#define CRYPTO_ECC_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "core/fwk.h"
#include "inc/minitls_errors.h"
#include "inc/minitls_config.h"
#include "crypto_prng.h"
#include "crypto_math.h"

typedef enum __crypto_ecc_curve_type
{
  sect163k1 = 1, sect163r1 = 2, sect163r2 = 3,
  sect193r1 = 4, sect193r2 = 5, sect233k1 = 6,
  sect233r1 = 7, sect239k1 = 8, sect283k1 = 9,
  sect283r1 = 10, sect409k1 = 11, sect409r1 = 12,
  sect571k1 = 13, sect571r1 = 14, secp160k1 = 15,
  secp160r1 = 16, secp160r2 = 17, secp192k1 = 18,
  secp192r1 = 19, secp224k1 = 20, secp224r1 = 21,
  secp256k1 = 22, secp256r1 = 23, secp384r1 = 24,
  secp521r1 = 25,
  //reserved = 0xFE00..0xFEFF,
  arbitrary_explicit_prime_curves = 0xFF01,
  arbitrary_explicit_char2_curves = 0xFF02,
  __crypto_ecc = 0xFFFF
} crypto_ecc_curve_type_t;

typedef struct __crypto_ecc_curve
{
  /** The size of the curve in octets */
  int size;

  /** Curve type */
  crypto_ecc_curve_type_t type;

  /** The prime that defines the field the curve is in (encoded in hex) */
  char *prime;

  /** The fields B param (hex) */
  char *B;

  /** The order of the curve (hex) */
  char *order;

  /** The x co-ordinate of the base point on the curve (hex) */
  char *Gx;

  /** The y co-ordinate of the base point on the curve (hex) */
  char *Gy;
} crypto_ecc_curve_t;

/** A point on a ECC curve, stored in Jacbobian format such that (x,y,z) => (x/z^2, y/z^3, 1) when interpretted as affine */
typedef struct {
    /** The x co-ordinate */
    //void *x;
    fp_int x;

    /** The y co-ordinate */
    //void *y;
    fp_int y;

    /** The z co-ordinate */
    //void *z;
    fp_int z;
} crypto_ecc_point_t;

typedef struct __crypto_ecc_public_key
{
  const crypto_ecc_curve_t* curve;

  crypto_ecc_point_t pubkey;
} crypto_ecc_public_key_t;

typedef struct __crypto_ecc_private_key
{
  crypto_ecc_public_key_t pub;
  fp_int privkey;
} crypto_ecc_private_key_t;

minitls_err_t crypto_ecc_curve_get(const crypto_ecc_curve_t** curve, crypto_ecc_curve_type_t type);
minitls_err_t crypto_ecc_ansi_x963_import(crypto_ecc_public_key_t* key, const crypto_ecc_curve_t* curve, const uint8_t* x963, size_t size);
minitls_err_t crypto_ecc_ansi_x963_export(const crypto_ecc_public_key_t* key, /*const crypto_ecc_curve_t* curve,*/ uint8_t* x963, size_t max_size, size_t* size);
minitls_err_t crypto_ecc_generate_key(crypto_ecc_private_key_t* key, const crypto_ecc_curve_t* curve, crypto_prng_t* prng);
size_t crypto_ecc_get_key_size_for_curve(const crypto_ecc_curve_t* curve);

const crypto_ecc_public_key_t* crypto_ecc_get_public_key(const crypto_ecc_private_key_t* private_key);

minitls_err_t crypto_ecc_dsa_check(const crypto_ecc_public_key_t* key, const uint8_t* hash, size_t hash_size, const uint8_t* signature, size_t signature_size);

minitls_err_t crypto_ecc_dh_generate_shared_secret(const crypto_ecc_private_key_t* private_key, const crypto_ecc_public_key_t* public_key, uint8_t* secret, size_t max_secret_size, size_t* secret_size);

#ifdef __cplusplus
}
#endif

#endif /* CRYPTO_ECC_H_ */
