/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file crypto_aes_128_cbc.c
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

#define __DEBUG__ 0
#ifndef __MODULE__
#define __MODULE__ "crypto_aes_128_cbc.c"
#endif


#include "core/fwk.h"
#include "inc/minitls_errors.h"

#include "crypto_aes_128_cbc.h"

minitls_err_t crypto_aes_128_cbc_encrypt(crypto_aes_128_t* cipher, buffer_t* initialization_vector, buffer_t* buffer)
{
  uint8_t* text = buffer_current_read_position(buffer);
  size_t size = buffer_length(buffer);

  uint8_t* previous = buffer_current_read_position(initialization_vector);

  if( (size % AES_128_BLOCK_SIZE) || (buffer_length(initialization_vector) != AES_128_BLOCK_SIZE) )
  {
    return MINITLS_ERR_WRONG_ALIGNMENT_FOR_CIPHER;
  }

  uint8_t xored_input[AES_128_BLOCK_SIZE]; //Xored input

  while(size > 0)
  {
    for(int i = 0; i < AES_128_BLOCK_SIZE / sizeof(uint32_t); i++)
    {
      ((uint32_t*)xored_input)[i] = ((uint32_t*)previous)[i] ^ ((uint32_t*)text)[i];
    }

    crypto_aes_128_encrypt(cipher, xored_input, text);

    previous = text; //Save previous ciphertext to use as initialization vector

    text += AES_128_BLOCK_SIZE;
    size -= AES_128_BLOCK_SIZE;
  }

  return MINITLS_OK;
}

minitls_err_t crypto_aes_128_cbc_decrypt(crypto_aes_128_t* cipher, buffer_t* initialization_vector, buffer_t* buffer)
{
  uint8_t* text = buffer_current_read_position(buffer);
  size_t size = buffer_length(buffer);

  if( (size % AES_128_BLOCK_SIZE) || (buffer_length(initialization_vector) != AES_128_BLOCK_SIZE) )
  {
    return MINITLS_ERR_WRONG_ALIGNMENT_FOR_CIPHER;
  }

  uint8_t xored_output[AES_128_BLOCK_SIZE]; //Xored input
  uint8_t previous[AES_128_BLOCK_SIZE*2];

  memcpy(previous, buffer_current_read_position(initialization_vector), AES_128_BLOCK_SIZE);

  while(size > 0)
  {
    //Save cipher text (to use as initialization vector)
    memcpy(previous + AES_128_BLOCK_SIZE, text, AES_128_BLOCK_SIZE);

    crypto_aes_128_decrypt(cipher, text, xored_output);

    for(int i = 0; i < AES_128_BLOCK_SIZE / sizeof(uint32_t); i++)
    {
      ((uint32_t*)text)[i] = ((uint32_t*)previous)[i] ^ ((uint32_t*)xored_output)[i];
    }

    memmove(previous, previous + AES_128_BLOCK_SIZE, AES_128_BLOCK_SIZE);

    text += AES_128_BLOCK_SIZE;
    size -= AES_128_BLOCK_SIZE;
  }

  return MINITLS_OK;
}

