/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file TLSSocket.cpp
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

#define __DEBUG__ 0//4
#ifndef __MODULE__
#define __MODULE__ "TLSSocket.cpp"
#endif

#include "core/fwk.h"
#include "TLSSocket.h"


#include "tls/minitls.h"
#include "tls/tls_socket.h"

/** Instantiate a TLS socket
 *
 */
TLSSocket::TLSSocket(MiniTLS* pMiniTLS) : m_pMiniTLS(pMiniTLS)
{

}

TLSSocket::~TLSSocket()
{

}

/** Initialize socket
 * \return MINITLS_OK on success, MINITLS_ERR_* error code otherwise
 */
minitls_err_t TLSSocket::init()
{
  int ret = tls_socket_init(&m_sock, &m_pMiniTLS->m_minitls, m_writeBuf, sizeof(m_writeBuf), m_readBuf, sizeof(m_readBuf));
  if(ret)
  {
    ERR("Could not init socket: error %d", ret);
    return ret;
  }
  return MINITLS_OK;
}


/** Connect to server
 * \param hostname server to connect to
 * \param port port to connect to
 * \param timeout timeout in ms
 * \return MINITLS_OK on success, MINITLS_ERR_* error code otherwise
 */
minitls_err_t TLSSocket::connect(const char* hostname, uint16_t port, int timeout)
{
  minitls_err_t ret = tls_socket_connect(&m_sock, hostname, port, timeout);
  if(ret)
  {
    return ret;
  }
  return MINITLS_OK;
}

/** Read data from server
 * \param buf buffer to read bytes to
 * \param minLength minimum number of bytes to read (will block before this number of bytes are read)
 * \param maxLength maximum number of bytes to read
 * \param readLength will receive actual number of bytes read
 * \param timeout timeout in ms
 * \return MINITLS_OK on success, MINITLS_ERR_* error code otherwise
 */
minitls_err_t TLSSocket::read(uint8_t* buf, size_t minLength, size_t maxLength, size_t* readLength, int timeout)
{
  if(minLength > maxLength)
  {
    return MINITLS_ERR_PARAMETERS;
  }

  *readLength = 0;

  minitls_err_t ret;
  while( *readLength < minLength )
  {
    ret = tls_socket_flush_read(&m_sock, timeout);
    if(ret == MINITLS_ERR_SOCKET_CLOSED)
    {
      return MINITLS_OK;
    }
    else if(ret)
    {
      return ret;
    }

    size_t bytesRead = 0;
    ret = tls_socket_read(&m_sock, buf + *readLength, maxLength - *readLength, &bytesRead);
    if(ret)
    {
      return ret;
    }

    *readLength += bytesRead;
  }

  return MINITLS_OK;
}

/** Write data to server
 * \param buf buffer to write bytes from
 * \param length number of bytes to write
 * \param writtenLength will receive actual number of bytes written
 * \param timeout timeout in ms
 * \return MINITLS_OK on success, MINITLS_ERR_* error code otherwise
 */
minitls_err_t TLSSocket::write(uint8_t* buf, size_t length, size_t* writtenLength, int timeout)
{
  *writtenLength = 0;

  minitls_err_t ret;
  while( *writtenLength < length )
  {
    size_t bytesToWrite = length - *writtenLength;
    size_t bytesWritten = 0;
    ret = tls_socket_write(&m_sock, buf + *writtenLength, bytesToWrite, &bytesWritten);
    if(ret == MINITLS_ERR_SOCKET_CLOSED)
    {
      return MINITLS_OK;
    }
    else if(ret)
    {
      return ret;
    }

    *writtenLength += bytesWritten;

    //Flush only if needed
    if( bytesWritten < bytesToWrite )
    {
      ret = tls_socket_flush_write(&m_sock, timeout);
      if(ret)
      {
        return ret;
      }
    }
  }

  return MINITLS_OK;
}

/** Call to ensure transmission of bytes written using write
 * \param timeout timeout in ms
 * \return MINITLS_OK on success, MINITLS_ERR_* error code otherwise
 */
minitls_err_t TLSSocket::flush(int timeout)
{
  minitls_err_t ret = tls_socket_flush_write(&m_sock, timeout);
  if(ret)
  {
    return ret;
  }
  return MINITLS_OK;
}

/** Tear down TLS connection and close socket
 * \return MINITLS_OK on success, MINITLS_ERR_* error code otherwise
 */
minitls_err_t TLSSocket::close()
{
  minitls_err_t ret = tls_socket_close(&m_sock);
  if(ret)
  {
    return ret;
  }
  return MINITLS_OK;
}

