/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file MiniTLS.cpp
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

#define __DEBUG__ 0//4
#ifndef __MODULE__
#define __MODULE__ "MiniTLS.cpp"
#endif

#include "MiniTLS.h"

#include "core/fwk.h"

#include "inc/minitls_config.h"
#include "tls/minitls.h"
#include "crypto/crypto_prng.h"
#include "crypto/crypto_ecc.h"
#include "crypto/crypto_rsa.h"

/** Create MiniTLS instance
 *
 * */
MiniTLS::MiniTLS()
{

}

MiniTLS::~MiniTLS()
{

}

/** Initialize MiniTLS library
 * This function will initialize the Pseudo Random Number Generator and the MiniTLS library
 */
void MiniTLS::init()
{
  crypto_prng_init(&m_prng, NULL); //TODO add mutex support
  minitls_init(&m_minitls, &m_prng);
}

/** Feed the Pseudo Random Number Generator with random seed data
 * \param buffer data to feed
 * \param length the buffer's length
 */
void MiniTLS::feedPRNG(uint8_t* buffer, size_t length)
{
  crypto_prng_feed(&m_prng, buffer, length);
}

/** Add a public certificate
 * The certificate must be x509-formatted
 * The key must be x963-formatted (ECC) or PKCS1-formatted (RSA)
 *  \param cert certificate
 *  \param certSize size of the certificate in bytes
 *  \param pubKey public key (can point within the certificate)
 *  \param pubKeySize public key's size
 *  \return MINITLS_OK on success, MINITLS_ERR_* error code otherwise
 *  \note only one certificate supported at the moment
 */
minitls_err_t MiniTLS::addCertificate(const uint8_t* cert, size_t certSize, const uint8_t* pubKey, size_t pubKeySize)
{
  
  minitls_err_t ret;
#if CRYPTO_ECC  
  const crypto_ecc_curve_t* curve;
  ret = crypto_ecc_curve_get(&curve, secp192r1);
  if(ret)
  {
    ERR("Unsupported elliptic curve");
    return ret;
  }

  ret = crypto_ecc_ansi_x963_import(&m_cert.public_key.ecc, curve, pubKey, pubKeySize);
  if(ret)
  {
    ERR("Error %d while decoding key", ret);
    return ret;
  }
#elif CRYPTO_RSA
  ret = crypto_rsa_pkcs1_import(&m_cert.public_key.rsa, pubKey, pubKeySize);
  if(ret)
  {
    ERR("Error %d while decoding key", ret);
    return ret;
  }
#else
#error
#endif

  m_cert.certificate = cert;
  m_cert.certificate_size = certSize;

  ret = minitls_certificate_add(&m_minitls, &m_cert);
  if(ret)
  {
    ERR("Error %d while registering certificate", ret);
    return ret;
  }

  return MINITLS_OK;
}


